/**
 * BEGIN FLOCK GPL
 *
 * Copyright Flock Inc. 2005-2009
 * http://flock.com
 *
 * This file may be used under the terms of the
 * GNU General Public License Version 2 or later (the "GPL"),
 * http://www.gnu.org/licenses/gpl.html
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * END FLOCK GPL
 */

"use strict";

CU.import("resource:///modules/FlockDateFormatter.jsm");
CU.import("resource:///modules/FlockXMLUtils.jsm");

// Bind $jQ to jQuery $ selector
var $jQ = jQuery.noConflict();

// Helper function to build DOM nodes.
var NodeBuilder = {
  // Define Namespaces
  HTML_NS: "http://www.w3.org/1999/xhtml",
  XUL_NS: "http://www.mozilla.org/keymaster/gatekeeper/there.is.only.xul",

  elementWithClass: function nodeBuilder_elementWithClass(aNodeName, aClass, aNS) {
    if (aNodeName) {
      var element = document.createElementNS((aNS) ? aNS : this.HTML_NS,
                                             aNodeName);
      if (aClass) {
        element.setAttribute("class", aClass);
      }
      return element;
    }
    return null;
  }
};

/******************************************************************************
 * The Facebook Chat API, for interacting with the Facebook servers.
 ******************************************************************************/

function FacebookChatApi(aLogger, aWD) {
  this._logger = aLogger;
  this._wd = aWD;

  this._nsJSON = CC["@mozilla.org/dom/json;1"].createInstance(CI.nsIJSON);

  this._incomingChatURL = this._wd.getLocalizedString(
          "facebookchat",
          "incomingChatURL",
          "http://0.%channel%.facebook.com/x/%nocache%/false/p_%uid%=%msgid%");

  // From nsIXMLHttpRequest.idl
  // Note: These cause a redefinition error when defined globally, but end up
  // undefined when actually needed. :(
  this._XHR_READYSTATE_UNINITIALIZED = 0;
  this._XHR_READYSTATE_LOADING = 1;
  this._XHR_READYSTATE_LOADED = 2;
  this._XHR_READYSTATE_INTERACTIVE = 3;
  this._XHR_READYSTATE_COMPLETED = 4;
}

// Perform a call to the Facebook servers.
FacebookChatApi.prototype.doCall =
function fbcApi_doCall(aUrl, aMethod, aParams, aFormat, aFlockListener) {
  var inst = this;
  var error;
  var xhr = CC["@mozilla.org/xmlextras/xmlhttprequest;1"]
            .createInstance(CI.nsIXMLHttpRequest);
  xhr.QueryInterface(CI.nsIXMLHttpRequest);

  xhr.onreadystatechange = function doCall_onrsc(aEvt) {
    // Is the request complete?
    if (xhr.readyState == inst._XHR_READYSTATE_COMPLETED) {
      inst._logger.debug("doCall_onrsc() status: " + xhr.status);
      // Was the request successful?
      if (Math.floor(xhr.status/100) == 2) {
        inst._logger.debug("\n===== begin xhr response =====\n"
                           + xhr.responseText.substring(0, 4096)
                           + "\n====== end xhr response ======");

        // For JSON responses, inspect for an error code.
        if (aFormat === "JSON") {
          var responseText = xhr.responseText.replace(/^[^{]+/, "");
          var responseJS = inst._nsJSON.decode(responseText);
          if (responseJS.error === 0) {
            aFlockListener.onSuccess(responseJS, arguments.callee.name);
          } else {
            error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
            error.serviceErrorCode = responseJS.error;
            error.serviceErrorString = responseJS.errorSummary + ": "
                                     + responseJS.errorDescription;
            switch (parseInt(responseJS.error, 10)) {
              case 1357001:   // Not Logged In
                error.errorCode = CI.flockIError.PHOTOSERVICE_USER_NOT_LOGGED_IN;
                break;
              case 1357004:   // Authentication Failure
                error.errorCode = CI.flockIError.PHOTOSERVICE_USER_NOT_LOGGED_IN;
                break;
              default:
                error.errorCode = CI.flockIError.PHOTOSERVICE_UNKNOWN_ERROR;
                break;
            }
            aFlockListener.onError(error, arguments.callee.name);
          }
        } else {
          aFlockListener.onSuccess(xhr.responseText, arguments.callee.name);
        }

      } else {
        // HTTP errors (0 for connection lost)
        error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
        error.errorCode = xhr.status;
        aFlockListener.onError(error, arguments.callee.name);
      }
    }
  };

  var paramList = "";
  var url = aUrl;
  var postvars = null;
  for (var p in aParams) {
    if (paramList.length) {
      paramList += "&";
    }
    paramList += p + "=" + encodeURIComponent(aParams[p]);
  }
  if (aMethod === "POST") {
    postvars = paramList;
  } else {
    url = url + "?" + paramList;
  }
  this._logger.debug("CALL: '" + url + "', '" + postvars + "'");
  xhr.mozBackgroundRequest = true;
  xhr.open(aMethod, url, true);

  if (aFormat == "JSON") {
    xhr.setRequestHeader("Content-Type", "application/x-www-form-urlencoded");
    xhr.overrideMimeType("text/txt");
  }
  xhr.send(postvars);
};

// Start the Comet loop that waits for incoming chat messages.
/*
Chat Incoming Flow:

  1. Request message ID "-1" (an invalid message)
  2. Server responds with valid current message ID
  3. Request current message ID, and wait
  4. Server will respond in about 60 seconds (or less) with either of:
      - "continue"    No message          Do nothing
      - "msg"         Message arrived     Process the message, increment ID
  5. Go to 3

Requests are made to:

  http://[ANYTHING].channel[NUM].facebook.com/x/[ANYTHING]/false/p_[UID]=[SEQ]
  http://0.channel06.facebook.com/x/0/false/p_123456789=4177168544

  See: http://coderrr.wordpress.com/2008/05/06/facebook-chat-api/
*/
FacebookChatApi.prototype.startIncomingChat =
function fbcApi_startIncomingChat(aUID, aChannel, aFlockListener) {
  var inst = this;

  // The message ID to be requested; "-1" will retrieve the valid value.
  this._chatMsgID = -1;
  this._incomingChatCancelled = false;

  var requestUrl;
  var responseText;
  var responseJS;

  // Create the XmlHttpRequest for the Comet loop.
  this._cometXhr = CC["@mozilla.org/xmlextras/xmlhttprequest;1"]
             .createInstance(CI.nsIXMLHttpRequest);
  this._cometXhr.QueryInterface(CI.nsIXMLHttpRequest);
  this._cometXhr.mozBackgroundRequest = true;

  // Convenience function to build the request URL.
  function buildChatRequestUrl(aMsgId) {
    var url = inst._incomingChatURL.replace("%channel%", aChannel)
                                   .replace("%uid%", aUID)
                                   .replace("%nocache%", new Date().getTime())
                                   .replace("%msgid%", aMsgId);
    inst._logger.debug("incomingChat requestUrl: '" + url + "'");
    return url;
  }

  function incomingChat_onrsc(aEvt) {
    // Is the request complete?
    if (inst._cometXhr &&
        inst._cometXhr.readyState == inst._XHR_READYSTATE_COMPLETED)
    {
      inst._logger.debug("incomingChat_onrsc status: " + inst._cometXhr.status);
      // Was the request successful?
      if (Math.floor(inst._cometXhr.status/100) == 2) {
        inst._logger.debug("\n===== begin xhr response =====\n"
                           + inst._cometXhr.responseText
                           + "\n====== end xhr response ======");

        // Retrieve the response object from the JSON.
        responseText = inst._cometXhr.responseText.replace(/^[^{]+/, "");
        responseJS = inst._nsJSON.decode(responseText);
        inst._logger.debug("incomingChat_onrsc(): '" + responseText + "'");
        inst._logger.debug("incomingChat_onrsc(): '" + responseJS.t + "'");
        switch (responseJS.t) {
          case "error":
            // Request the initial sequence number again.
            inst._chatMsgID = -1;
            break;
          case "refresh":
            if (responseJS.seq === undefined) {
              // Comments in Pidgin say that when no sequence number is
              // provided, the session or post_form_id is no longer valid.
              // Log out, and make user log in again.
              gFacebookChat.svcLogout();
            } else {
              // Pick up the current message sequence number from the response.
              inst._chatMsgID = parseInt(responseJS.seq, 10);
            }
            break;
          case "continue":
            // Timeout with no message received; do nothing, repeat.
            break;
          case "msg":
            // Woohoo! We received a message.
            aFlockListener.onSuccess(responseJS, null);
            // Request the new sequence number, and repeat.
            inst._chatMsgID++;
            break;
        }

        // Build the updated request and repeat the process, reusing xhr.
        // http://radio.javaranch.com/pascarello/2006/03/31/1143817890773.html
        if (!inst._incomingChatCancelled) {
          requestUrl = buildChatRequestUrl(inst._chatMsgID);
          inst._cometXhr.open("GET", requestUrl, true);
          inst._cometXhr.onreadystatechange = incomingChat_onrsc;
          inst._cometXhr.send(null);
        }

      } else {
        // HTTP errors (0 for connection lost)
        var error = CC["@flock.com/error;1"].createInstance(CI.flockIError);
        error.errorCode = inst._cometXhr.status;
        aFlockListener.onError(error, null);
      }
    }
  }

  // Build the initial request and kick-start the process.
  requestUrl = buildChatRequestUrl(this._chatMsgID);
  this._cometXhr.open("GET", requestUrl, true);
  this._cometXhr.onreadystatechange = incomingChat_onrsc;
  this._cometXhr.send(null);
};

// Terminate the Comet loop that waits for incoming chat messages.
FacebookChatApi.prototype.stopIncomingChat =
function fbcApi_stopIncomingChat() {
  this._logger.debug("stopIncomingChat() begin");
  this._incomingChatCancelled = true;
  if (this._cometXhr) {
    // Abort if the XHR exists.
    this._cometXhr.abort();
  }
  this._cometXhr = null;
  this._logger.debug("stopIncomingChat() end");
};

/******************************************************************************
 * Typing Notification Object
 ******************************************************************************/

// Constructor
function FbTypingTimer(aParent, aUID) {
  this._parent = aParent;
  this._toUID = aUID;
  this._timerCount = 0;
  this._timer = CC["@mozilla.org/timer;1"].createInstance(CI.nsITimer);
  // We delay before sending a "started typing" message, because short messages
  // will be sent immediately anyway.
  this._timer.initWithCallback(this,
                               2 * 1000,   // every 2 seconds
                               CI.nsITimer.TYPE_REPEATING_SLACK);
}

// Reset the "stopped typing" timer..
FbTypingTimer.prototype.reset =
function fbtt_reset() {
  // Do not reset unless the timer has already fired once.
  if (this._timerCount) {
    // Reset to 1, so the "started" message is not re-sent.
    this._timerCount = 1;
  }
};

// Abort without sending the "Stopped typing" message.
FbTypingTimer.prototype.cancel =
function fbtt_cancel() {
  this._timer.cancel();
  this._timer = null;
  this._parent._fbttInstance = null;
};

// nsITimerCallback: Send "(Started|Stopped) typing" messages.
FbTypingTimer.prototype.notify =
function fbtt_notify(aTimer) {
  switch (this._timerCount) {
    case 0:   // Started typing (On Facebook, 2-second delay)
      this._parent.sendTypingNotification({"to": this._toUID, "typ": "1"});
      break;
    case 5:   // Stopped typing (On Facebook, 10-second delay)
      this._parent.sendTypingNotification({"to": this._toUID, "typ": "0"});
      this.cancel();
      break;
  }
  this._timerCount++;
};


/******************************************************************************
 * Emotions/Smilies Object
 ******************************************************************************/

// Constructor
function FbEmotionsParser() {
  this._emotionsMap = null;
  this._regexp = null;
  this.init();
}

// Initialize the emotions parser
FbEmotionsParser.prototype.init =
function fbep_init() {
  // Array of currently supported emotions
  var emotions = ["smile", "frown", "tongue", "grin", "gasp", "wink",
                  "glasses", "sunglasses", "grumpy", "unsure", "cry", "devil",
                  "angel", "kiss", "heart", "kiki", "squint", "confused",
                  "upset", "pacman", "colonthree"];

  // Setup a map of regular expressions to the type of emotion
  this.emotionsMap = {":-)": ["\\:\\-\\)", "smile"],
                      ":)": ["\\:\\)", "smile"],
                      ":]": ["\\:\\]", "smile"],
                      "=)": ["=\\)", "smile"],
                      ":-(": ["\\:\\-\\(", "frown"],
                      ":(": ["\\:\\(", "frown"],
                      ":[": ["\\:\\[", "frown"],
                      "=(": ["=\\(", "frown"],
                      ":-P": ["\\:\\-P", "tongue"],
                      ":P": ["\\:P", "tongue"],
                      ":-p": ["\\:\\-p", "tongue"],
                      ":p": ["\\:p", "tongue"],
                      "=P": ["=P", "tongue"],
                      ":-D": ["\\:\\-D", "grin"],
                      ":D": ["\\:D", "grin"],
                      "=D": ["=D", "grin"],
                      ":-O": ["\\:\\-O", "gasp"],
                      ":O": ["\\:O", "gasp"],
                      ":-o": ["\\:\\-o", "gasp"],
                      ":o": ["\\:o", "gasp"],
                      ";-)": ["\\;\\-\\)", "wink"],
                      ";)": ["\\;\\)", "wink"],
                      "8-)": ["8\\-\\)", "glasses"],
                      "8)": ["8\\)", "glasses"],
                      "B-)": ["B\\-\\)", "glasses"],
                      "B)": ["B\\)", "glasses"],
                      "8-|": ["8\\-\\|", "sunglasses"],
                      "8|": ["8\\|", "sunglasses"],
                      "B-|": ["B\\-\\|", "sunglasses"],
                      "B|": ["B\\|", "sunglasses"],
                      ">:(": [">\\:\\(", "grumpy"],
                      ">:-(": [">\\:\\-\\(", "grumpy"],
                      ":/": ["\\:/", "unsure"],
                      ":-/": ["\\:\\-/", "unsure"],
                      ":\\": ["\\:\\\\", "unsure"],
                      ":-\\": ["\\:\\-\\\\", "unsure"],
                      ":'(": ["\\:'\\(", "cry"],
                      "3:)": ["3\\:\\)", "devil"],
                      "3:-)": ["3\\:\\-\\)", "devil"],
                      "O:)": ["O\\:\\)", "angel"],
                      "O:-)": ["O\\:\\-\\)", "angel"],
                      ":-*": ["\\:\\-\\*", "kiss"],
                      ":*": ["\\:\\*", "kiss"],
                      "<3": ["<3", "heart"],
                      "^_^": ["\\^_\\^", "kiki"],
                      "-_-": ["\\-_\\-", "squint"],
                      "o.O": ["o\\.O", "confused"],
                      "O.o": ["O\\.o", "confused"],
                      ">:O": [">\\:O", "upset"],
                      ">:-O": [">\\:\\-O", "upset"],
                      ">:o": [">\\:o", "upset"],
                      ">:-o": [">\\:\\-o", "upset"],
                      ":v": ["\\:v", "pacman"],
                      ":|]": ["\\:\\|\\]", "robot"],
                      ":3": ["\\:3", "colonthree"] };

  // Create one regular expression to detect the occurrence of any emotion
  var emotionsRegExpArray = [];
  for (var key in this.emotionsMap) {
    emotionsRegExpArray.push(this.emotionsMap[key][0]);
  }

  // Join all regular expressions to create one joint regular expression
  var emtionsRegExpString = '(?:^|\\s|\'|"|\\.)('
                          + emotionsRegExpArray.join('|')
                          + ')(?:\\s|\'|"|\\.|,|!|\\?|<br>|$)';

  this._regex = new RegExp(emtionsRegExpString);
};

FbEmotionsParser.prototype.parseText =
function fbep_parseText(aText) {
  var node = NodeBuilder.elementWithClass("div", null);

  var parsedString = aText;
  while (true) {
    // Determine if the string contains any emotions
    var matchResult = this._regex.exec(parsedString);

    if (!matchResult || !matchResult.length) {
      break;
    }

    var match = matchResult[1];
    var matchIndex = parsedString.indexOf(match);

    // Recreate the string and replace with the emotion
    var emotionKey = this.emotionsMap[match][1];

    // Grab anything before the emotion text
    var emotionStrPrefix = parsedString.substring(0, matchIndex);
    if (emotionStrPrefix) {
      node.appendChild(document.createTextNode(emotionStrPrefix));
    }

    var emotionDiv =
      NodeBuilder.elementWithClass("div", "emotion " + emotionKey);
    node.appendChild(emotionDiv);

    // Continue with the remaining text
    parsedString = parsedString.substring(matchIndex + match.length);
  }
  if (parsedString) {
    node.appendChild(document.createTextNode(parsedString));
  }
  return node;
};

FbEmotionsParser.prototype.parseNode =
function fbep_parse(aNode) {
  // Determine if the node and its children contain any emotions
  var node;
  for (var i = 0; i < aNode.childNodes.length; i++) {
    node = aNode.childNodes[i];
    if (node && node.hasChildNodes()) {
      node = this.parseNode(node);
    }
    if (node.nodeType == 3) {
      // Text node, parse emotions
      var newNode = this.parseText(node.nodeValue);
      var parentNode = node.parentNode;
      parentNode.removeChild(node);
      parentNode.appendChild(newNode);
    }
  }
  return aNode;
};


/******************************************************************************
 * Request Header Parser
 ******************************************************************************/

// Constructor
function FbHeaderParser() {
  this._obsService = CC["@mozilla.org/observer-service;1"]
                     .getService(CI.nsIObserverService);
  this._obsService.addObserver(this, "http-on-examine-response", false);

  this._settingsURL = gFacebookChat.getWD()
                                   .getLocalizedString("facebookchat",
                                                       "settingsURL",
                                                       null);
}

FbHeaderParser.prototype.uninit =
function fbhp_uninit() {
  // Remove HTTP request observers
  this._obsService.removeObserver(this, "http-on-examine-response");
};

FbHeaderParser.prototype.observe =
function fbhp_observe(aSubject, aTopic, aData) {
  if (aTopic == "http-on-examine-response") {
    this.onExamineResponse(aSubject);
  }
};

FbHeaderParser.prototype.onExamineResponse =
function fbhp_onExamineResponse(aHttp) {
  var inst = gFacebookChat;

  var uri = aHttp.QueryInterface(CI.nsIHttpChannel).URI.asciiSpec;
  if (uri == this._settingsURL) {
    var postString = this.readPostTextFromRequest(aHttp);
    if (postString) {
      // Parse out to see whether or not we have a visibility change
      var keyValues = postString.split("&");
      var key;
      var value;
      var splitted;
      for each (var keyValue in keyValues) {
        splitted = keyValue.split("=");
        key = splitted[0];
        value = splitted[1];
        if (key == "visibility") {
          inst._logger.debug("POST data received: setting visibility to: "
                             + value);
          inst._chatLoggedIn = (value == "true") ? true : false;
          inst.refreshVisibility();
          return;
        }
      }
    }
  }
};

FbHeaderParser.prototype.readPostTextFromRequest =
function fbhp_readPostTextFromRequest(aRequest) {
  const NS_SEEK_SET = CI.nsISeekableStream.NS_SEEK_SET;
  try {
   var uploadChannel = aRequest.QueryInterface(CI.nsIUploadChannel);
   var uploadStream = uploadChannel.uploadStream;
    if (uploadStream) {
      var seekableStream = uploadStream.QueryInterface(CI.nsISeekableStream);
      if (seekableStream) {
        seekableStream.seek(NS_SEEK_SET, 0);
      }
      var text = this.readFromStream(uploadStream);
      if (seekableStream) {
        seekableStream.seek(NS_SEEK_SET, 0);
      }
      return text;
    }
  } catch (ex) {
    // Post data unavailable
  }
  return null;
};

FbHeaderParser.prototype.readFromStream =
function fbhp_readFromStream(aStream) {
  try {
    var inputStream = CC["@mozilla.org/binaryinputstream;1"]
                      .getService(CI.nsIBinaryInputStream);
    inputStream.setInputStream(aStream);
    var segments = [];
    for (var count = aStream.available(); count; count = aStream.available()) {
      segments.push(inputStream.readBytes(count));
    }
    return segments.join("");
  } catch (ex) {
    // Cannot read from stream
  }
  return null;
};

/******************************************************************************
 * Facebook Chat -- Implementation.
 ******************************************************************************/

var gFacebookChat = {
  FACEBOOK_CONTRACT_ID: "@flock.com/people/facebook;1",

  PREF_CHAT_ENABLED: "flock.service.facebook.chat.enabled",
  _isEnabled: true,

  DEFAULT_ANIMATION_DURATION: 600,
  DEFAULT_HIGHLIGHT_DURATION: 2500,

  BADGE_CHAT_X: -16,
  BADGE_CHAT_Y: 16,

  BADGE_NOTIFICATION_X: 1,
  BADGE_NOTIFICATION_Y: 20,

  // Properties File
  BUNDLE_URI: "chrome://flock/locale/services/facebookchat.properties",

  // Width of the overflow container
  _tabBarOverflowWidth: null,
  _tabBarIsAnimating: false,

  TAB_WIDTH: 140,
  STATUS_BAR_OFFSET: 400,

  // Max length a tab name before trucation
  TAB_NAME_MAX_LENGTH: 14,
  BUDDY_LIST_NAME_MAX_LENGTH: 18,
  PANEL_STATUS_MAX_LENGTH: 19,
  STATUS_MAX_LENGTH: 26,
  STATUS_MAX_LINK_LENGTH: 24,

  // Offset off window height and buddyList height
  BUDDY_LIST_Y_OFFSET: 230,
  BUDDY_HEIGHT: 30,
  BUDDY_HEIGHT_IDLE: 18,

  _Urls: {
    login: "http://facebook.com/login.php",
    home: "http://www.facebook.com/home.php",
    profile: "http://www.facebook.com/profile.php?id=",
    readUpdate: "http://www.facebook.com/ajax/presence/update.php",
    chatSettings: "http://www.facebook.com/ajax/chat/settings.php",
    chatSend: "http://www.facebook.com/ajax/chat/send.php",
    chatTyp: "http://www.facebook.com/ajax/chat/typ.php"
  },

  _emotionsParser: null,
  _headerParser: null,

  // Information from the login success page that we need to keep.
  _chatVisible: true,
  _chatIsHiding: false,

  _svcLoggedIn: false,
  _chatLoggedIn: false,
  _buddysLoaded: false,

  _loggedInUserInfo: null,
  _account: null,
  _UID: null,
  _matchedName: null,
  _channel: null,
  _postFormId: null,
  _lastestNotification: null,
  _notificationBadgeID: null,
  _lastOutgoingMsgClientTime: null,

  // Accounts Observer
  _accountsObserver: null,
  _obsService: null,

  // Flock user chat hash
  _userChatHash: {},
  _lastActiveChatPanelUID: null,
  _lastClosedUID: null,

  // Facebook active chats when facebook is first loaded
  _activeFacebookChats: {},

  // Avoid redundant initialization.
  _isInitialized: false,

  _resizeTime: null,
  _resizeInterval: null,
  _resizeHideUID: null,
  _browserWidth: null,
  _browserHeight: null,

  _tooltipMode: null
};


/******************************************************************************
 * Utility Functions
 ******************************************************************************/

gFacebookChat.__defineGetter__("chatPopupset",
function fbc_getchatPopupset() {
  return document.getElementById("chatPopupset");
});

gFacebookChat.__defineGetter__("badgePopupset",
function fbc_getbadgePopupset() {
  return document.getElementById("badgePopupset");
});

gFacebookChat.__defineGetter__("chatStatusBarPanel",
function fbc_getchatStatusBarPanel() {
  return document.getElementById("chatStatusBarPanel");
});

// Overflow container to handle case when too many tabs are on screen
gFacebookChat.__defineGetter__("tabBarOverflow",
function fbc_gettabBar() {
  return document.getElementById("tabBarOverflow");
});

gFacebookChat.__defineGetter__("tabBarContainer",
function fbc_gettabBar() {
  return document.getElementById("tabBarContainer");
});

// List of all friends
gFacebookChat.__defineGetter__("buddyList",
function fbc_getbuddyList() {
  return document.getElementById("buddyList");
});

// Online friends panel
gFacebookChat.__defineGetter__("buddyListPanel",
function fbc_getbuddyListPanel() {
  return document.getElementById("buddyListPanel");
});

// Online Friends tab
gFacebookChat.__defineGetter__("buddyListTab",
function fbc_getbuddyListTab() {
  return document.getElementById("buddyListTab");
});

// Chat Status Panel
gFacebookChat.__defineGetter__("chatStatusPanel",
function fbc_getchatStatusPanel() {
  return document.getElementById("chatStatusPanel");
});

// Chat Status Tab
gFacebookChat.__defineGetter__("chatStatusTab",
function fbc_getchatStatusTab() {
  return document.getElementById("chatStatusTab");
});

// Notifications Container within Notification Panel
gFacebookChat.__defineGetter__("notificationsContent",
function fbc_getNotificationsContent() {
  return document.getElementById("notificationsContent");
});

// Notifications Tab
gFacebookChat.__defineGetter__("notificationsTab",
function fbc_getnotificationsTab() {
  return document.getElementById("notificationsTab");
});

// Notifications Panel
gFacebookChat.__defineGetter__("notificationsPanel",
function fbc_getnotificationsPanel() {
  return document.getElementById("notificationsPanel");
});

// Tooltip Panel
gFacebookChat.__defineGetter__("tooltipPanel",
function fbc_gettooltipPanel() {
  return document.getElementById("tooltipPanel");
});

// Get all panels within the main panel grouping
gFacebookChat.getPanelGroupMembers =
function fbc_getPanelGroupMembers() {
  if (!this._panelGroupMembers) {
    this._panelGroupMembers = [this.buddyListPanel,
                               this.chatStatusPanel,
                               this.notificationsPanel];
  }
  return this._panelGroupMembers;
};

// Get all tabs within the main panel grouping
gFacebookChat.getTabGroupMembers =
function fbc_getTabGroupMembers() {
  if (!this._tabGroupMembers) {
    this._tabGroupMembers = [this.buddyListTab,
                             this.notificationsTab,
                             this.chatStatusTab];
  }
  return this._tabGroupMembers;
};

gFacebookChat.getWD =
function fbc_getWD() {
  if (!this._wd) {
    this._wd = CC["@flock.com/web-detective;1"]
               .getService(CI.flockIWebDetective);
    acUtils = CC["@flock.com/account-utils;1"]
              .getService(CI.flockIAccountUtils);
    acUtils.loadWebDetectiveFile("facebookchat.xml", "res/detect");
    //this._wd.loadFromChrome("chrome://facebookchat/content/facebookchat.xml");
  }
  return this._wd;
};

gFacebookChat.getCurrentWindow =
function fbc_getCurrentWindow() {
  return CC["@mozilla.org/appshell/window-mediator;1"]
         .getService(CI.nsIWindowMediator)
         .getMostRecentWindow("navigator:browser");
};

gFacebookChat.getLogger =
function fbc_getLogger() {
  if (!this._logger) {
    this._logger = CC["@flock.com/logger;1"]
                   .createInstance(CI.flockILogger);
    this._logger.init("facebookchat");
  }
  return this._logger;
};

gFacebookChat.getBundle =
function fbc_getBundle() {
  if (!this._bundle) {
    this._bundle = CC["@mozilla.org/intl/stringbundle;1"]
                   .getService(CI.nsIStringBundleService)
                   .createBundle(this.BUNDLE_URI);
  }
  return this._bundle;
};

gFacebookChat.getPrefs =
function fbc_getPrefs() {
  if (!this._prefs) {
    this._prefs = CC["@mozilla.org/preferences-service;1"]
                  .getService(CI.nsIPrefBranch);
  }
  return this._prefs;
};

gFacebookChat.getFacebookSvc =
function fbc_getFacebookSvc() {
  if (!this._fbService) {
    this._fbService = CC[this.FACEBOOK_CONTRACT_ID]
                      .getService(CI.flockIWebService);
  }
  return this._fbService;
};

gFacebookChat.getAccountsSvc =
function fbc_getAccountsSvc() {
  if (!this._acctsService) {
    this._acctsService = CC["@flock.com/account-service;1"]
                         .getService(CI.flockIAccountService);
  }
  return this._acctsService;
};

gFacebookChat.getBuddyFilterPlaceholder =
function fbc_getBuddyFilterPlaceholder() {
  if (!this._buddyFilterPlaceholder) {
    this._buddyFilterPlaceholder =
      this.getBundle()
          .GetStringFromName("facebookchat.buddyPanel.filter.placeholder");
  }
  return this._buddyFilterPlaceholder;
};

/******************************************************************************
 * Memeber Functions
 ******************************************************************************/

gFacebookChat.svcLogin =
function fbc_svcLogin(aAccount) {
  this._svcLoggedIn = true;
  this._account = aAccount;
  this._UID = aAccount.getParam("accountId");
  this.ensureAccountInfoInit();
  this.getParameters();
};

gFacebookChat.svcLogout =
function fbc_svcLogout() {
  this._logger.info("svcLogout");

  this.setChatStatus("error");

  // Close the chat status tab
  this.chatStatusPanel.hidePopup();
  this.chatStatusTab.removeAttribute("selected");

  // Reset variables
  this.chatLogout();
  this._svcLoggedIn = false;
  this._chatLoggedIn = false;
  this._buddysLoaded = false;
  this._account = null;
  this._channel = null;
  this._postFormId = null;
  this._UID = null;
  this._matchedName = null;
  this._lastestNotification = null;
  this._lastOutgoingMsgClientTime = null;

  this._activeFacebookChats = {};
  this._userChatHash = {};
  this._lastClosedUID = null;

  // Clear all relevant DOM nodes
  var removeNodes = [this.chatPopupset, this.badgePopupset,
                     this.tabBarContainer, this.buddyList];
  for each (var node in removeNodes) {
    this.clearChildrenForNode(node);
  }

  // Hide the tooltip panel
  this.hideTooltip();

  // Make sure all group panels and tabs have been reset
  this.resetGroupTabStates();
  this.resetGroupPanelStates();
};

/******************************************************************************
 * Chat Server Interaction
 ******************************************************************************/

// Retrieve the needed parameters to invoke a chat session
gFacebookChat.getParameters =
function fbc_getParameters() {
  // Indicate to the user that we are fetching the login state
  this.setChatStatus("loading");

  var inst = this;
  var getParametersListener = {
    onSuccess: function getParams_onSuccess(aSubject, aTopic) {
      inst._logger.debug("getParameters_onSuccess:");

      // Create RegExp from web detective strings
      var wd = inst.getWD();
      var channelRegExp = new RegExp(wd.getLocalizedString("facebookchat",
                                                           "channelRegExp",
                                                           null));
      var postFormIdRegExp =
        new RegExp(wd.getLocalizedString("facebookchat",
                                         "postFormIdRegExp",
                                          null));
      var chatLoggedInRegExp =
        new RegExp(wd.getLocalizedString("facebookchat",
                                         "chatLoggedInRegExp",
                                         null));
      var activeChatsRegExp =
        new RegExp(wd.getLocalizedString("facebookchat",
                                         "activeChatsRegExp",
                                         null), "g");

      inst._refreshInterval = parseInt(wd.getLocalizedString("facebookchat",
                                                             "buddyRefreshSeconds",
                                                             "180"),
                                       10);

      var nameRegExp =
        new RegExp(wd.getLocalizedString("facebookchat",
                                         "nameRegExp",
                                         null));

      // Perform matches
      var channelResults = aSubject.match(channelRegExp);
      var postFormIdResults = aSubject.match(postFormIdRegExp);
      var chatLoggedInResults = aSubject.match(chatLoggedInRegExp);
      var activeChatsResults = aSubject.match(activeChatsRegExp);
      var nameResults = aSubject.match(nameRegExp);

      // Grab the user's name if matched
      if (nameResults && nameResults.length >= 1) {
        inst._matchedName = nameResults[1];
        inst.refreshAccountInfo();
      }

      // Looks for any active chats the user already has open
      if (activeChatsResults && activeChatsResults.length >= 0) {
        var result, uid;
        for (var i = 0; i < activeChatsResults.length; i++) {
          // Retrieve UIDs and number of unreceived messages from facebook
          result = activeChatsResults[i];
          uid = result.substring(0, result.indexOf("\""));
          var msgCount = result.substring(result.lastIndexOf(":") + 1,
                                          result.length);
          if (uid && msgCount) {
            inst._activeFacebookChats[uid] = { uid: uid, msgCount: msgCount };
          }
        }
      }

      // Grab the required parameters for server communication
      if (channelResults && channelResults.length >= 0 &&
          postFormIdResults && postFormIdResults.length >= 1 &&
          chatLoggedInResults && chatLoggedInResults.length >= 1)
      {
        // Parameters found
        inst._channel = channelResults[0];
        inst._postFormId = postFormIdResults[1];
        inst._chatLoggedIn = (chatLoggedInResults[1] == "1") ? true : false;
        inst.setChatStatus(inst.getChatStatus());

        inst._logger.info("getParameters: found channel = " + inst._channel
                          + ", postFormId = " + inst._postFormId
                          + ", loggedIn = " + inst._chatLoggedIn);

        if (inst._chatLoggedIn) {
          // Get the buddy list if logged into Facebook Chat
          inst.chatLogin();
        }
        return;
      }
      inst.setChatStatus("offline");
    },
    onError: function getParams_onError(aFlockError, aTopic) {
      inst._logger.debug("getParams_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
    }
  };
  this._Api.doCall(this._Urls.home,
                   "POST",
                   {},
                   "TEXT",
                   getParametersListener);
};

/**
 * Set the user's visibility (online status)
 * @param flockIListener aFlockListener
 * @param Boolean aStatus - Online/Offline stautus of the user
 */
gFacebookChat.setVisibility =
function fbc_setVisibility(aFlockListener, aStatus) {
  // Can only send status once logged into Facebook
  if (!this._svcLoggedIn) {
    return;
  }

  var inst = this;
  var setVisibilityListener = {
    onSuccess: function setVisibility_onSuccess(aSubject, aTopic) {
      inst._logger.debug("setVisibility_onSuccess:" + aSubject);
      if (aFlockListener) {
        aFlockListener.onSuccess(aSubject, arguments.callee.name);
      }
    },
    onError: function setVisibility_onError(aFlockError, aTopic) {
      inst._logger.debug("setVisibility_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
      if (aFlockListener) {
        aFlockListener.onError(aFlockError, arguments.callee.name);
      }
    }
  };

  var params = {
    post_form_id: this._postFormId,
    post_form_id_source: "AsyncRequest",
    visibility: aStatus
  };
  this._Api.doCall(this._Urls.chatSettings,
                   "POST",
                   params,
                   "JSON",
                   setVisibilityListener);
};

gFacebookChat.chatLogout =
function fbc_chatLogout() {
  this._logger.debug("fbc_chatLogout()");

  // Cancel the Comet loop
  this._Api.stopIncomingChat();

  // Cancel the timer that refreshes the buddy list.
  if (this._buddyListTimer) {
    this._buddyListTimer.cancel();
    this._buddyListTimer = null;
  }
};

// Initialize the buddylist refresh timer.
gFacebookChat.resetBuddyListTimer =
function fbc_resetBuddyListTimer() {
  if (this._buddyListTimer) {
    this._buddyListTimer.cancel();
    this._buddyListTimer.initWithCallback(this,
                                          this._refreshInterval * 1000,
                                          CI.nsITimer.TYPE_REPEATING_SLACK);
  }
};

// Log in to the chat server.
gFacebookChat.chatLogin =
function fbc_chatLogin() {
  this._logger.debug("fbc_chatLogin()");

  // Retrieve the buddy list
  this.getChatData();

  // Tee up a timer to do periodic buddy list refreshes.
  this._buddyListTimer = CC["@mozilla.org/timer;1"].createInstance(CI.nsITimer);
  this.resetBuddyListTimer();

  // Start the Facebook chat incoming stream.
  var inst = this;
  var incomingChatListener = {
    onSuccess: function incomingChat_onSuccess(aSubject, aTopic) {
      inst.getLogger().info("incomingChatListener_onSuccess");
      var messages = aSubject.ms;
      for each (var message in messages) {
        inst._logger.debug("Incoming message type: '" + message.type + "'");

        // Look for any focus, unfocus or close events from Facebook
        if (message.type && !message.from) {
          var type = message.type;
          if (message.id) {
            var UID = message.id;
            switch (type) {
              case "focus_chat":
                if (!inst._chatVisible) {
                  // If the chat is currently hidden save this as the most
                  // recently focused chat
                  inst._lastActiveChatPanelUID = UID;
                }
                inst.ensureTabInit(inst._userChatHash[UID], false);
                break;

              case "close_chat":
                if (inst._lastClosedUID != UID) {
                  inst.closeChat(null, UID);
                }
                break;
            }
          } else {
            switch (type) {
              case "unfocus_chat":
                if (!inst._resizeHideUID) {
                  inst._lastActiveChatPanelUID = null;
                }
                break;
              case "notification":
                // A new notification has been received, retrieve it.
                inst.getChatData();
                break;
              case "notifications_read":
                // The notifications have been read, clear our badge if present
                inst.removeNotificationTabBadge();
                break;
            }
          }
          // Done with this message, continue to the next
          continue;
        }

        if (message && message.from) {
          var fromUID = message.from;
          var toUID = message.to;
          var tab;

          // Grab the UID of the respective chat partner
          var buddyUID = inst.getBuddyUID(fromUID, toUID);
          if (!buddyUID) {
            continue;
          }

          if (!message.msg) {
            // Does not contain any message text, is this a typing message?
            if (fromUID && message.type == "typ") {
              tab = inst.getTabForUID(buddyUID);
              if (tab) {
                if (parseInt(message.st, 10) == 1) {
                  tab.setAttribute("typing", "true");
                } else {
                  tab.removeAttribute("typing");
                }
              }
            }
            // No message data, try the next one.
            continue;
          }

          if (fromUID && fromUID == inst._UID &&
              inst._lastOutgoingMsgClientTime &&
              inst._lastOutgoingMsgClientTime == message.msg.clientTime)
          {
            // If the message was just sent, it has already been added to the
            // conversation, account for it in the chat history
            if (inst._userChatHash[toUID]) {
              inst._userChatHash[toUID].history[message.msg.time] = true;
            }
          } else {
            // Since we are recieving events from other instances of chat don't
            // assume this message is being sent to the logged in user
            if (!inst._userChatHash[fromUID]) {
              inst._userChatHash[fromUID] = { uid: fromUID,
                                              name: message.from_name };
              inst.ensureChatHistoryInitForUID(fromUID);
            }
            var fromBuddy = inst._userChatHash[fromUID];

            if (!inst._userChatHash[toUID]) {
              inst._userChatHash[toUID] = { uid: toUID,
                                            name: message.to_name };
              inst.ensureChatHistoryInitForUID(toUID);
            }
            var toBuddy = inst._userChatHash[toUID];

            var isTabInitialized =
              (inst._userChatHash[buddyUID].tabID != null);

            // Open the panel if the tab has not been initialized and the chat
            // bar is visible and this is the most current browser window
            // Note: windowState == 2 means a minimized window for Win OS, Macs
            // do not change this properly unfortunately
            var shouldOpenPanel = (!isTabInitialized &&
                                    inst._chatVisible &&
                                    window.top == inst.getCurrentWindow() &&
                                    window.windowState != 2);
            inst.ensureTabInit(fromBuddy, shouldOpenPanel);

            // Make sure the typing attribute is removed
            tab = inst.getTabForUID(buddyUID);
            tab.removeAttribute("typing");

            // Make sure we haven't already recieved this message
            if (!inst._userChatHash[buddyUID].history[message.msg.time]) {

              // If the chat panel is closed and the tab was already
              // initialized highlight the tab to indicate a new message
              // has been recieved
              if (isTabInitialized && !inst.isChatPanelOpenForUID(buddyUID)) {
                // Highlight the tab to indicate a new incoming message
                inst.selectTabWithState(tab, "highlight");

                  // Increment the new message count
                if (inst._userChatHash[buddyUID].unreadMsgCount < 99) {
                  inst._userChatHash[buddyUID].unreadMsgCount++;
                }

                // Add a notification badge to the tab
                inst.ensureNotificationBadgeInit(tab,
                  inst._userChatHash[buddyUID].unreadMsgCount);
              }

              inst.addMessageToPanel(fromBuddy,
                                     toBuddy,
                                     message.msg.text,
                                     message.msg.time,
                                     "incoming");
            }
          }
        }
      }
      // Write the received chat message to the window.
      /*
      {"t":"msg"
       "c":"p_620501201",
       "ms":[ {"type":"typ",
               "st":1,                  // 1 = Started, 0 = Stopped
               "from":504815839,
               "to":620501201},
              {"type":"focus_chat",     // Opened local chat window.
               "id":504815839}
              {"type":"unfocus_chat"}   // Minimized local chat window.
              {"type":"close_chat",     // Closed my local chat window.
               "id":504815839}
              {"type":"msg",
                "msg":{"text":"asasdjdsajklasd",
                       "time":1237844323328,
                       "clientTime":1237844322025,
                       "msgID":"1414420682"},
                "from":504815839,       // These fields are reversed as needed
                "to":620501201,
                "from_name":"Mary Smith",
                "to_name":"Bob Fisher",
                "from_first_name":"Mary",
                "to_first_name":"Bob"}
             ]
      }
      */
    },
    onError: function incomingChat_onError(aFlockError, aTopic) {
      // Connection lost, server error, whatever... What shall we do?
    }
  };
  this._Api.startIncomingChat(this._UID, this._channel, incomingChatListener);
};

// Timer callback to refresh the buddy list and notifications.
gFacebookChat.notify =
function fbc_notify(aTimer) {
  this.getChatData();
  if (this.sentMessageCount > 0) {
    this._metrics.report("FacebookChat-MessagesSent", this.sentMessageCount);
    this.sentMessageCount = 0;
  }
};

// Retrieve the user's buddy list and notifications from the chat server.
gFacebookChat.getChatData =
function fbc_getChatData() {
  if (!this._chatLoggedIn) {
    return;
  }
  var inst = this;
  var buddyListListener = {
    onSuccess: function buddyList_onSuccess(aSubject, aTopic) {
      inst._logger.debug("buddyList_onSuccess:" + aSubject);
      // Parse the buddy list
      inst.parseBuddyList(aSubject.payload.buddy_list);
      // Parse the user's notifications
      inst.parseNotifications(aSubject.payload.notifications);
    },
    onError: function buddyList_onError(aFlockError, aTopic) {
      inst._logger.debug("buddyList_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
    }
  };

  var params = {
    buddy_list: "1",
    notifications: "1", 
    force_render: "true",
    post_form_id: this._postFormId,
    user: this._UID
  };
  this._Api.doCall(this._Urls.readUpdate,
                   "POST",
                   params,
                   "JSON",
                   buddyListListener);
};

gFacebookChat.parseBuddyList =
function fbc_parseBuddyList(aBuddyList) {
  if (aBuddyList.listChanged || !this._buddysLoaded) {
    // Sync our buddy list with the received list.
    var userInfos = aBuddyList.userInfos;
    var avail = aBuddyList.nowAvailableList;
    var buddy;
    var uid;

    for (uid in userInfos) {
      // Our own account appears in the buddy list; skip it.
      if (uid == this._UID) {
        continue;
      }
      // Update buddy particulars.
      if (this._userChatHash[uid]) {
        // Update a buddy we already have in our list.
        this._userChatHash[uid].name = userInfos[uid].name;
        this._userChatHash[uid].firstName = userInfos[uid].firstName;
        this._userChatHash[uid].thumbSrc = userInfos[uid].thumbSrc;
        this._userChatHash[uid].status = userInfos[uid].status;
        this._userChatHash[uid].statusTime = userInfos[uid].statusTime;
        this._userChatHash[uid].statusTimeRel = userInfos[uid].statusTimeRel;
        this._userChatHash[uid].available = "C";  // offline
        this._userChatHash[uid].sync = "updated";
      } else {
        // Add this buddy to our list.
        buddy = userInfos[uid];
        buddy.uid = uid;
        buddy.available = "C";  // offline; yes, these happen.
        buddy.tabID = null;
        buddy.chatPanelID = null;
        buddy.badgeID = null;
        buddy.unreadMsgCount = 0;
        buddy.sync = "new";
        this._userChatHash[uid] = buddy;
      }

      // Keep track of chat message history
      this.ensureChatHistoryInitForUID(uid, true);

      // Update availability.
      if (avail[uid]) {
        if (avail[uid].i) {
            this._logger.debug("buddy '" + uid + "' is IDLE");
            this._userChatHash[uid].available = "B"; // idle
        } else {
            this._logger.debug("buddy '" + uid + "' is ONLINE");
            this._userChatHash[uid].available = "A"; // online
        }
      }
    }

    // Make the UI match the internal buddy list.
    for each (buddy in this._userChatHash) {
      switch (buddy.sync) {
        case "":
          // Buddy should be removed from list and from UI.
          buddy.sync = "removed";
          this.removeBuddy(buddy);
          break;
        case "new":
          this.addBuddy(buddy);
          buddy.sync = "";
          break;
        case "updated":
          // Buddy should be updated.
          this.removeBuddy(buddy);
          this.addBuddy(buddy);
          buddy.sync = "";
          break;
      }
    }

    if (!this._buddysLoaded) {
      // Open the tabs in the order they were sequenced
      for (uid in this._activeFacebookChats) {
        this.ensureTabInit(this._userChatHash[uid], false);
        if (this._activeFacebookChats[uid].msgCount &&
            this._activeFacebookChats[uid].msgCount > 0)
        {
          var tab = this.getTabForUID(uid);
          var msgCount = this._activeFacebookChats[uid].msgCount;

          // Highlight the tab to indicate a unviewed messages
          this.selectTabWithState(tab, "highlight");

          // Facebook limits the message count to 99
          msgCount = (msgCount > 99) ? 99 : msgCount;

          // Add a notification badge to the tab
          this.addNotificationBadge(tab, msgCount);

          this._userChatHash[uid].unreadMsgCount = msgCount;
        }
      }
    }

    // Set the buddy count.
    this.setBuddyCount(aBuddyList.availableCount);
    this.resizeBuddyListContainer();
    this._buddysLoaded = true;
  }
  /* Properties:
    { "error":0,
      "errorSummary":"",
      "errorDescription":"No error.",
      "payload":
      { "buddy_list":
        { "listChanged":true,
          "availableCount":1,
          "nowAvailableList":
          { "UID1":
            {"i":false}
          },
          "wasAvailableIDs":[],
          "userInfos":
          { "UID1":
            { "name":"Buddy 1",
              "firstName":"Buddy",
              "thumbSrc":"http:\/\/static.ak.fbcdn.net\/pics\/q_default.gif",
              "status":null,
              "statusTime":0,
              "statusTimeRel":""
            },
            "UID2":
            { "name":"Buddi 2",
              "firstName":"Buddi",
              "thumbSrc":"http:\/\/static.ak.fbcdn.net\/pics\/q_default.gif",
              "status":null,
              "statusTime":0,
              "statusTimeRel":""
            }
          },
          "forcedRender":true
        },
        "time":1209560380000
      }
    }
  */
};

// Retrieve the user's notifications
gFacebookChat.parseNotifications =
function fbc_parseNotifications(aNotifications) {
  var inst = gFacebookChat;
  // The markup returned by facebook contains various javascript scope which
  // we don't have, clean it up first.
  function _sanitizeNode(aNode) {
    if (!aNode) {
      return null;
    }
    for (var i = 0; i < aNode.childNodes.length; i++) {
      var child = aNode.childNodes[i];
      if (child && child.hasChildNodes()) {
        child = _sanitizeNode(child);
      }
      child.onmouseover = null;
      child.onmouseout = null;
      child.onclick = null;
    }
    return gFacebookChat.convertLinks(aNode, "href");
  }

  function _xmlDecodeMarkup(aMarkup) {
    return aMarkup.replace(/&raquo;/g, "&#187;")
                  .replace(/&laquo;/g, "&#171;")
                  .replace(/&rsaquo;/g, "&#8250;")
                  .replace(/&lsaquo;/g, "&#8249;")
                  .replace(/&rdquo;/g, "\"")
                  .replace(/&ldquo;/g, "\"")
                  .replace(/&gt;/g, "&#62;")
                  .replace(/&lt;/g, "&#60;");
  }

  if (aNotifications) {
    var payloadLatestNotification = aNotifications.latest_notif;

    // Determine if we need to update our notification content
    if (!this._lastestNotification ||
        (this._lastestNotification &&
         payloadLatestNotification > this._lastestNotification))
    {
      // Clear out any existing nodes
      this.clearChildrenForNode(this.notificationsContent);

      // Mark the most recent notification recieved
      this._lastestNotification = payloadLatestNotification;

      if (aNotifications.countNew !== undefined) {
        if (aNotifications.countNew > 0) {
          // Add a notification badge to the tab
          this.ensureNotificationBadgeInit(this.notificationsTab,
                                           aNotifications.countNew);
        } else if (aNotifications.countNew == 0 &&
                   this.notificationsBadgeID)
        {
          // There are no new notifications, remove the badge
          this.removeNotificationTabBadge();
        }
      }

      if (aNotifications.markup) {
        var node = NodeBuilder.elementWithClass("div", null);
        node.innerHTML = _xmlDecodeMarkup(aNotifications.markup);
        node = _sanitizeNode(node);

        // Add our sanitized markup
        this.notificationsContent.appendChild(node);
      }

      if (aNotifications.inboxCount) {
        // Also append the how many new inbox messages the user has
        var webmailNode = NodeBuilder.elementWithClass("div", "notification");
        var webmailIcon = NodeBuilder.elementWithClass("div", "icon");
        webmailNode.appendChild(webmailIcon);

        // Body
        var webmailBody = NodeBuilder.elementWithClass("div", "body");
        webmailBody.setAttribute("id", "webmailBody");

        var inboxOnclickFunction =
        function inboxOnclickFunction_onClick(aEvent) {
          // Open Facebook's Inbox
          inst.openURL(inst.getWD().getLocalizedString("facebookchat",
                                                       "inboxURL",
                                                       null));
          inst._metrics.report("FacebookChat-ViewInbox", null);
        };

        var inboxText = NodeBuilder.elementWithClass("a", "inboxText");
        inboxText.innerHTML =
          this.getBundle()
              .GetStringFromName("facebookchat.notificationsPanel.viewInbox");
        inboxText.onclick = inboxOnclickFunction;

        var inboxCount = NodeBuilder.elementWithClass("a", "inboxCount");
        inboxCount.innerHTML = "(" + aNotifications.inboxCount + ")";
        inboxCount.onclick = inboxOnclickFunction;

        webmailBody.appendChild(inboxText);
        webmailBody.appendChild(inboxCount);
        webmailNode.appendChild(webmailBody);

        this.notificationsContent.appendChild(webmailNode);
      }
    }
  }
  /*
  {"payload":{"time":1237910677000,
              "notifications":{"countNew":0,
                               "count":5,
                               "app_names":{"2372251387":"Flock Browser",
                                            "2305272732":"Photos"},
                               "latest_notif":1237571057,
                               "latest_read_notif":1237571057,
                               "markup":""
                               "inboxCount":9,
                               "inboxDropdownMarkup":""}
             }
   }
  */
};

// Retrieve the chat history given a UID
gFacebookChat.getChatHistory =
function fbc_getChatHistory(aPanel, aUID) {
  var inst = this;
  var chatHistoryListener = {
    onSuccess: function chatHistoryListener_onSuccess(aSubject, aTopic) {
      inst.getLogger().debug("chatHistoryListener_onSuccess:" + aSubject);

      // Only load the history once
      inst._userChatHash[aUID].historyLoaded = true;

      var conversation =
        aPanel.getElementsByClassName("conversationContent")[0];

      // Add clear history link
      var clearHistory = NodeBuilder.elementWithClass("div", "clearHistory");
      clearHistory.onclick = function clearHistory_onClick() {
        inst.clearHistoryForUID(aUID, conversation);
      };
      var clearHistoryText =
        inst.getBundle().GetStringFromName("facebookchat.panel.history.clear");
      inst.setTextForNode(clearHistory, clearHistoryText);
      conversation.parentNode.insertBefore(clearHistory, conversation);

      // Reverse the array as will will be inserting them in reverse order
      // to allow for new messages to be appended to the end during a chat
      // history fetch
      var history = aSubject.payload.history.reverse();
      for each (var message in history) {
        conversation.setAttribute("loading", "true");
        var buddyUID = inst.getBuddyUID(message.to, message.from);
        if (message.msg &&
            message.msg.text &&
            buddyUID &&
            !inst._userChatHash[buddyUID].history[message.time])
        {
          if (!inst._userChatHash[buddyUID]) {
            // This buddy's info has not been setup yet, the payload gives us
            // this data, so init the buddy first
            var userInfo = aSubject.payload.userInfo;
            inst._userChatHash[buddyUID].name = userInfo.name;
            inst._userChatHash[buddyUID].firstName = userInfo.firstName;
            inst._userChatHash[buddyUID].thumbSrc = userInfo.thumbSrc;
            inst._userChatHash[buddyUID].status = userInfo.status;
            inst._userChatHash[buddyUID].statusTime = userInfo.statusTime;
            inst._userChatHash[buddyUID].statusTimeRel = userInfo.statusTimeRel;
            inst.ensureChatHistoryInitForUID(buddyUID);
          }
          // Add the message history to the panel
          inst.addMessageToPanel(inst._userChatHash[message.from],
                                 inst._userChatHash[message.to],
                                 message.msg.text,
                                 message.time,
                                 "history");
        }
      }
      conversation.removeAttribute("loading");

      // Ensure we are viewing the most recent message
      conversation.parentNode.scrollTop = conversation.scrollHeight;
      /*
      {
      "payload":{"history":[{"msg":{"text":""},
                             "from":814230018,
                             "to":507691193,
                             "time":1238091602909,
                             "type":"msg"}],
                 "userInfo":{"name":"",
                             "firstName":"",
                             "thumbSrc":"",
                             "status":"",
                             "statusTime":1237937061,
                             "statusTimeRel":""},
               }
      }
      */
    },
    onError: function chatHistoryListener_onError(aFlockError, aTopic) {
      inst.getLogger().debug("chatHistoryListener_onError: "
                             + aFlockError.serviceErrorCode + ": "
                             + aFlockError.serviceErrorString);
    }
  };

  this._Api.doCall(this.getWD().getLocalizedString("facebookchat",
                                                   "chatHistoryURL",
                                                   null),
                   "GET",
                   { id: aUID },
                   "JSON",
                   chatHistoryListener);
};

// Send a chat message to a buddy.
gFacebookChat.sendChatMessage =
function fbc_sendChatMessage(aFriendUID, aText) {
  if (!this._chatLoggedIn) {
    return;
  }
  var inst = this;

  var chatSendListener = {
    onSuccess: function chatSend_onSuccess(aSubject, aTopic) {
      inst._logger.debug("chatSend_onSuccess:");
      inst.sentMessageCount++;
    },
    onError: function chatSend_onError(aFlockError, aTopic) {
      inst._logger.debug("chatSend_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
      var bundle = inst.getBundle();
      var errorMsg;
      switch (aFlockError.serviceErrorCode) {
        case "1356002":
          // Not Online: You are not online to chat.
          errorMsg =
            bundle.GetStringFromName("facebookchat.panel.error.selfOffline");
          inst.addMessageToPanel(inst._loggedInUserInfo,
                                 inst._userChatHash[aFriendUID],
                                 errorMsg,
                                 inst._lastOutgoingMsgClientTime,
                                 "errorFeedback");
          break;
        case "1356003":
          // Send destination not online: <NAME> is no longer online.
          errorMsg =
            bundle.formatStringFromName("facebookchat.panel.error.buddyOffline",
                                        [inst._userChatHash[aFriendUID].name,
                                        aText],
                                        2);
          inst.addMessageToPanel(inst._loggedInUserInfo,
                                 inst._userChatHash[aFriendUID],
                                 errorMsg,
                                 inst._lastOutgoingMsgClientTime,
                                 "errorFeedback");
          break;
        case "1356004":
          // An unexpected error occurred. Please try again.
          errorMsg =
            bundle.GetStringFromName("facebookchat.panel.error.unexpected");
          inst.addMessageToPanel(inst._loggedInUserInfo,
                                 inst._userChatHash[aFriendUID],
                                 errorMsg,
                                 inst._lastOutgoingMsgClientTime,
                                 "errorFeedback");
          break;
      }
    }
  };

  // Keep track of this so we can tell if incoming messages from the
  // chat listener have already been added to our message panel
  this._lastOutgoingMsgClientTime = new Date().getTime();

  var params = {
    msg_text: aText,
    msg_id: this._lastOutgoingMsgClientTime,
    client_time: this._lastOutgoingMsgClientTime,
    to: aFriendUID,
    post_form_id: this._postFormId
  };
  this._Api.doCall(this._Urls.chatSend,
                   "POST",
                   params,
                   "JSON",
                   chatSendListener);
};

// Sync windows with facebook
gFacebookChat.syncEventWithFacebook =
function fbc_syncEventWithFacebook(aEventParams) {
  var inst = this;

  var syncEventListener = {
    onSuccess: function syncEvent_onSuccess(aSubject, aTopic) {
      inst._logger.debug("syncEvent_onSuccess:");
    },
    onError: function syncEvent_onError(aFlockError, aTopic) {
      inst._logger.debug("syncEvent_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
    }
  };

  var params = {
    post_form_id_source: "AsyncRequest",
    post_form_id: this._postFormId
  };
  // Add any additional parameters
  for (var key in aEventParams) {
    params[key] = aEventParams[key];
  }
  this._Api.doCall(this._Urls.chatSettings,
                   "POST",
                   params,
                   "JSON",
                   syncEventListener);
};

// Send notification that we are typing (or not).
gFacebookChat.sendTypingNotification =
function fbc_sendTypingNotification(aEventParams) {
  var inst = this;

  var typingNotifyListener = {
    onSuccess: function typingNotify_onSuccess(aSubject, aTopic) {
      inst._logger.debug("typingNotify_onSuccess:");
    },
    onError: function typingNotify_onError(aFlockError, aTopic) {
      inst._logger.debug("typingNotify_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
    }
  };

  var params = {
    post_form_id_source: "AsyncRequest",
    post_form_id: this._postFormId
  };
  // Add any additional parameters
  for (var key in aEventParams) {
    params[key] = aEventParams[key];
  }
  this._Api.doCall(this._Urls.chatTyp,
                   "POST",
                   params,
                   "JSON",
                   typingNotifyListener);
};

// Clears the chat history for a given UID
gFacebookChat.clearHistoryForUID =
function fbc_clearHistoryForUID(aUID, aConversationDiv) {
  var inst = this;

  var clearHistoryListener = {
    onSuccess: function clearHistory_onSuccess(aSubject, aTopic) {
      inst._logger.debug("clearHistory_onSuccess:");
      inst.clearChildrenForNode(aConversationDiv);
    },
    onError: function clearHistory_onError(aFlockError, aTopic) {
      inst._logger.debug("clearHistory_onError: " + aFlockError.serviceErrorCode
                         + ": " + aFlockError.serviceErrorString);
    }
  };

  var params = {
    clear_history_id: aUID,
    post_form_id_source: "AsyncRequest",
    post_form_id: this._postFormId
  };
  this._Api.doCall(this._Urls.chatSettings,
                   "POST",
                   params,
                   "JSON",
                   clearHistoryListener);
};

// Marks the notifications as read
gFacebookChat.markNotificationsRead =
function fbc_markNotificationsRead() {
  var inst = this;

  // Remove the badge from the notifications tab
  inst.removeNotificationTabBadge();

  var markNotificationsReadListener = {
    onSuccess: function markNotificationsRead_onSuccess(aSubject, aTopic) {
      inst._logger.debug("markNotificationsRead_onSuccess:");
    },
    onError: function markNotificationsRead_onError(aFlockError, aTopic) {
      inst._logger.debug("markNotificationsRead_onError: "
                         + aFlockError.serviceErrorCode + ": "
                         + aFlockError.serviceErrorString);
    }
  };

  var params = {
    post_form_id_source: "AsyncRequest",
    post_form_id: this._postFormId
  };
  this._Api.doCall(this.getWD().getLocalizedString("facebookchat",
                                                   "notificationsReadURL",
                                                   null),
                   "POST",
                   params,
                   "JSON",
                   markNotificationsReadListener);
};

/******************************************************************************
 * Observers / Listeners
 ******************************************************************************/

gFacebookChat.isURIFacebook =
function fbc_isURIFacebook(aURI) {
  var facebookHost = this.getWD().getLocalizedString("facebookchat",
                                                     "facebookHost",
                                                     null);

  var isHostFacebook = false;
  try {
    isHostFacebook = new RegExp(facebookHost).test(aURI.host);
  } catch (aException) {
    // When visiting a custom Flock page (about:page) this fails gracefully
    // catch so we can move on with the logic
  }
  return isHostFacebook;
};

/**
 * void onLocationChange(in nsIWebProgress aProgress,
 *                       in nsIRequest aRequest,
 *                       in nsIURI aURI);
 * @see nsIWebProgressListener#onLocationChange
 */
gFacebookChat.onLocationChange =
function fbc_onLocationChange(aProgress, aRequest, aURI) {
  var inst = gFacebookChat;
  var isVisitingFacebook = inst.isURIFacebook(aURI);

  // Make sure the tooltip panel is hidden onLocationChange
  inst.hideTooltip();

  if (!inst.isStatusBarVisible()) {
    // For a window without a status bar disregard showing panels
    return;
  }

  if (inst._chatVisible && isVisitingFacebook) {
    inst._chatIsHiding = true;
    inst.toggleChatVisible(false);
    inst._chatIsHiding = false;

    // Hide any notification badges
    for (var i = 0; i < inst.badgePopupset.childNodes.length; i++) {
      if (inst.badgePopupset.childNodes[i].state == "open") {
        inst.badgePopupset.childNodes[i].hidePopup();
      }
    }
    return;
  }

  if (!inst._chatVisible && !isVisitingFacebook) {
    inst.toggleChatVisible(true);
    // Reopen any panels that were open before hiding the chat
    inst.reopenActivePanels();

    // Open any notification badges
    for (var i = 0; i < inst.badgePopupset.childNodes.length; i++) {
      var badge = inst.badgePopupset.childNodes[i];
      if (badge && badge.state == "closed") {
        var type = badge.getAttribute("type");
        var tab = (type == "chat")
                ? inst.getTabForUID(badge.getAttribute("uid"))
                : inst.notificationsTab;
        if (tab) {
          var position = inst.getBadgePosition(type);
          badge.openPopup(tab, "before_end", position.x, position.y);
        }
      }
    }
    return;
  }

  if (inst.chatStatusBarPanel.getAttribute("status") == "error") {
    // Check to see if there is an athenticated account
    inst.checkForAuthenticatedAccount();
  }
};

// nsIWebProgressListener#onStateChange stub
gFacebookChat.onStateChange =
function fbc_onStateChange(aWebProgress,
                           aRequest,
                           aStateFlags,
                           aStatus)
{
};

// nsIWebProgressListener#onProgressChange stub
gFacebookChat.onProgressChange =
function fbc_onProgressChange(aWebProgress,
                              aRequest,
                              aCurSelfProgress,
                              aMaxSelfProgress,
                              aCurTotalProgress,
                              aMaxTotalProgress)
{
};

// nsIWebProgressListener#onStatusChange stub
gFacebookChat.onStatusChange =
function fbc_onStatusChange(aWebProgress,
                            aRequest,
                            aStatus,
                            aMessage)
{
};

// nsIWebProgressListener#onSecurityChange stub
gFacebookChat.onSecurityChange =
function fbc_onSecurityChange(aWebProgress,
                              aRequest,
                              aState)
{
};

gFacebookChat.toggleChatVisible =
function fbc_toggleChatVisible(aState) {
  this.getLogger().debug("toggleChatVisible: " + aState);
  this._chatVisible = aState;

  if (aState) {
    this.setChatStatus(this.getChatStatus());
  } else {
    this.setChatStatus("hidden");
    // Close chat panels, while saving the current state of any open chats
    this.resetAllChats();
    this.resetGroupTabStates();
    this.resetGroupPanelStates();
  }
};

gFacebookChat.checkForFacebookAuthChange =
function fbc_checkForFacebookAuthChange(aAccount) {
  var logger = this.getLogger();
  // Is this a facebook account?
  if (aAccount instanceof CI.flockIWebServiceAccount) {
    if (aAccount.getService().contractId == this.FACEBOOK_CONTRACT_ID) {
      if (aAccount.isAuthenticated()) {
        // Login the account
        logger.info("Facebook onAccountAuthChange: login");
        this.svcLogin(aAccount);
      } else {
        // The user has just logged out of this account
        logger.info("Facebook onAccountAuthChange: logout");
        this.svcLogout();
      }
    }
  }
};

/**
 * When an auth change occurs check to see if the account in question belongs
 * to the Facebook Service.
 */
gFacebookChat.addAccountsObserver =
function fbc_addAccountsObserver() {
  var inst = gFacebookChat;

  // Setup the observer
  inst._accountsObserver = {
    onAccountAuthenticated:
    function accountsObs_onAccountAuthenticated(aAccount) {
      inst.checkForFacebookAuthChange(aAccount);
    },
    onAccountRemoved:
    function accountsObs_onAccountRemoved(aAccount) {
      // Stub function, no-op
      inst.hideTooltip();
    },
    onAccountUnauthenticated:
    function accountsObs_onAccountUnauthenticated(aAccount) {
      inst.checkForFacebookAuthChange(aAccount);
    },
    onAccountRefreshing:
    function accountsObs_onAccountRefreshing(aAccount) {
      // Stub function, no-op
    },
    onAccountNoLongerRefreshing:
    function accountsObs_onAccountNoLongerRefreshing(aAccount) {
      // Stub function, no-op
    }
  };
  inst.getAccountsSvc().addObserver(inst._accountsObserver);
};

/******************************************************************************
 * User Interface
 ******************************************************************************/

// Close all panels, and reset all tab selection states
gFacebookChat.resetAllChats =
function fbc_resetAllChats() {
  this.minimizeAllChats(null);
  this.resetChatTabsState();
};

// Minimize all chat panels being displayed
gFacebookChat.minimizeAllChats =
function fbc_minimizeAllChats(aNoHidePanel) {
  this.getLogger().info("minimizeAllChats");
  this._lastActiveChatPanelUID = null;

  var panels = this.chatPopupset.childNodes;
  for (var i = 0; i < panels.length; i++) {
    if (panels[i] == aNoHidePanel) {
      continue;
    }
    if (panels[i].state == "open") {
      panels[i].hidePopup();
      this._lastActiveChatPanelUID = panels[i].getAttribute("uid");
    }
  }
};

// Minimize a specified chat panel
gFacebookChat.minimizeChat =
function fbc_minimizeChat(aUID) {
  this.getLogger().info("minimizeChat for uid: " + aUID);

  if (this._userChatHash[aUID] && this._userChatHash[aUID].chatPanelID) {
    // Hide the respective chat window
    var chatPanel =
      document.getElementById(this._userChatHash[aUID].chatPanelID);
    chatPanel.hidePopup();
  }
  this.resetChatTabsState();
};

// Close a specified chat panel and it's respective tab
gFacebookChat.closeChat =
function fbc_closeChat(aEvent, aUID) {
  this.getLogger().info("closeChat for uid: " + aUID);
  this._lastClosedUID = aUID;

  // Prevent this event from propagating to a tab click
  if (aEvent) {
    aEvent.stopPropagation();
  }

  // Close any panels before repositioning the tabs
  this.resetAllChats();

  if (this._userChatHash[aUID]) {
    if (this._userChatHash[aUID].chatPanelID) {
      // Close the respective chat window
      var chatPanel =
        document.getElementById(this._userChatHash[aUID].chatPanelID);
      chatPanel.parentNode.removeChild(chatPanel);
      this._userChatHash[aUID].chatPanelID = null;
      this._userChatHash[aUID].historyLoaded = false;

      // Reset the history
      this.ensureChatHistoryInitForUID(aUID, true);
    }
    if (this._userChatHash[aUID].tabID) {
      // Close the respective tab
      var chatTab =
        document.getElementById(this._userChatHash[aUID].tabID);
      chatTab.parentNode.removeChild(chatTab);
      this._userChatHash[aUID].tabID = null;

      // Make sure the overflow scroll left is set properly
      this.ensureScrollIsWithinBounds();

      // Check for tab sizing on screen
      this.determineTabControllerState();

      // Sync on tab close, as this will always happen when closing a chat;
      // a chat panel on the other hand cannot exist yet when closing
      this.syncEventWithFacebook({"close_chat": aUID});
    }
    // Remove the notification badge
    this.removeNotificationBadgeForUID(aUID);
  }

  // Open any panels previously open before the tab close occured
  this.reopenActivePanels();
};

// Load a Facebook profile in a new tab
gFacebookChat.loadProfileUrl =
function fbc_loadProfileUrl(aEvent, aUID) {
  this.getLogger().info("loadProfileUrl for uid: " + aUID);

  if (aEvent.button == 2) {
    // Right click
    return;
  }

  var url = gFacebookChat._Urls.profile + aUID;
  this.openURL(url);
};

// Make sure the message history has been initialized for a given UID
// or reset it if aForceReset is set to true
gFacebookChat.ensureChatHistoryInitForUID =
function fbc_ensureChatHistoryInitForUID(aUID, aForceReset) {
  if (!this._userChatHash[aUID].history || aForceReset) {
    this._userChatHash[aUID].history = {};
    this._userChatHash[aUID].history.lastHeader = {};
    this._userChatHash[aUID].history.lastHeader.uid = null;
    this._userChatHash[aUID].history.lastHeader.time = null;
  }
};

gFacebookChat.addMessageToPanel =
function fbc_addMessageToPanel(aFromBuddy, aToBuddy, aText, aTime, aType) {
  var fromUID = aFromBuddy.uid;
  var panelUID;
  var panelInitBuddy;
  var sentFromLoggedInUser = false;
  if (this._UID == fromUID) {
    // Message being sent
    panelUID = aToBuddy.uid;
    panelInitBuddy = aToBuddy;
    sentFromLoggedInUser = true;
  } else {
    // Message is being recieved
    panelUID = fromUID;
    panelInitBuddy = aFromBuddy;
  }

  var isHistory = false;
  var showHeaderIfApplicable = true;
  switch (aType) {
    case "errorFeedback":
      showHeaderIfApplicable = false;
      break;

    case "history":
      isHistory = true;
      break;
  }

  this.ensureChatPanelInit(panelInitBuddy);

  // Get a handle on the panel conversation and append the message
  var chatPanel =
    document.getElementById(this._userChatHash[panelUID].chatPanelID);

  var conversation = chatPanel.getElementsByClassName("conversationContent")[0];
  if (conversation) {
    var messageContainer =
      NodeBuilder.elementWithClass("div", "messageContainer");
    messageContainer.setAttribute("type", aType);

    // Make sure the message history has been initialized
    this.ensureChatHistoryInitForUID(panelUID, false);

    var lastHeaderTime = this._userChatHash[panelUID].history.lastHeader.time;
    var lastHeaderUID = this._userChatHash[panelUID].history.lastHeader.uid;
    var ellapsedTimeBetweenHeaders = (lastHeaderTime)
                                   ? (Math.abs(aTime - lastHeaderTime) / 1000) / 60
                                   : null;
    var referenceElement;

    // If consecutive messages have been sent from the same user in a 1
    // minute timeframe supress the name header and group the messages
    if (showHeaderIfApplicable &&
        ((!lastHeaderUID && !lastHeaderTime) ||
         (lastHeaderUID && lastHeaderUID != fromUID) ||
          ((lastHeaderUID && lastHeaderUID == fromUID) &&
           (ellapsedTimeBetweenHeaders && ellapsedTimeBetweenHeaders > 1))))
    {
      var messageHeader =
        NodeBuilder.elementWithClass("div", "chat_message_header");
      messageHeader.setAttribute("type", aType);
      var clearFix = NodeBuilder.elementWithClass("div", "clearFix");

      var messageFrom = NodeBuilder.elementWithClass("div", "message_from");
      messageFrom.innerHTML = aFromBuddy.name;
      if (sentFromLoggedInUser) {
        messageFrom.setAttribute("from", "self");
      } else {
        // This is another user, link to their profile
        messageFrom.onclick = function _loadProfile(aEvent) {
          gFacebookChat.loadProfileUrl(aEvent, panelUID);
        };
      }

      var messageTime = NodeBuilder.elementWithClass("div", "time");
      messageTime.innerHTML = this.formatMessageDate(new Date(aTime));

      messageHeader.appendChild(messageFrom);
      messageHeader.appendChild(messageTime);
      messageHeader.appendChild(clearFix);

      if (isHistory && conversation.hasChildNodes()) {
        // Insert the node before the message when from history
        referenceElement = conversation.firstChild;
        conversation.insertBefore(messageHeader, referenceElement);
      } else {
        messageContainer.appendChild(messageHeader);
      }

      // Keep track of the last message header time
      this._userChatHash[panelUID].history.lastHeader.uid = fromUID;
      this._userChatHash[panelUID].history.lastHeader.time = aTime;
    }

    // Parse out any links
    var parsedMessage = this._account.getService().parseStatus(document, aText);
    // Parse any emotions from the DOM node rendered from parseStatus
    parsedMessage = this._emotionsParser.parseNode(parsedMessage);
    // Convert any links for proper loading in browser scope
    parsedMessage = this.convertLinks(parsedMessage, "origURL");

    var message = NodeBuilder.elementWithClass("div", "chat_message");
    message.appendChild(parsedMessage);

    messageContainer.appendChild(message);

    if (isHistory && conversation.hasChildNodes()) {
      referenceElement = conversation.firstChild.nextSibling;
      conversation.insertBefore(messageContainer, referenceElement);
    } else {
      conversation.appendChild(messageContainer);
    }

    // Add this message to the conversation history
    this._userChatHash[panelUID].history[aTime] = true;

    // Make sure we have scrolled to the bottom
    conversation.parentNode.scrollTop = conversation.scrollHeight;
  }
};

// See if the chat panel has been initialized
gFacebookChat.isChatPanelOpenForUID =
function fbc_isChatPanelOpenForUID(aUID) {
  if (this._userChatHash[aUID] && this._userChatHash[aUID].chatPanelID) {
    var panel = document.getElementById(this._userChatHash[aUID].chatPanelID);
    if (panel) {
      return (panel.state == "open");
    }
  }
  return false;
};

// Make sure the panel has been initialized
gFacebookChat.ensureChatPanelInit =
function fbc_ensureChatPanelInit(aUser) {
  var uid = aUser.uid;
  if (this._userChatHash[uid] &&
      !this._userChatHash[uid].chatPanelID)
  {
    this.createChatPanel(aUser);
  }
};

gFacebookChat.ensureTabInit =
function fbc_ensureTabInit(aUser, aShouldOpenPanel) {
  var UID = aUser.uid;
  var tabAddPerformed = false;

  if (UID == this._UID) {
    // Never setup a tab for the logged in user
    return;
  }

  if (!this._userChatHash[UID]) {
    // Has the user been initialized
    this._userChatHash[UID] = aUser;
    this.ensureChatHistoryInitForUID(UID, true);
  }

  if (!this._userChatHash[UID].tabID) {
    // Make sure the tab is also initialized
    this.getLogger().debug("Starting new chat.");
    this.addTab(this._userChatHash[UID]);
    tabAddPerformed = true;
  }

  if (aShouldOpenPanel) {
    if (!tabAddPerformed) {
      // When addTab() is called ensureChatTabIsVisible() gets called,
      // make sure we only call this once
      this.ensureChatTabIsVisible(UID);
    }

    var chatTab =
      document.getElementById(this._userChatHash[UID].tabID);
    this.tabClick(chatTab, this._userChatHash[UID], true);
  }
};

// Update the status message and time to reflect any changes
gFacebookChat.chatPanelOnPopupShowing =
function fbc_chatPanelOnPopupShowing(aEvent) {
  var inst = gFacebookChat;

  var panel = aEvent.target;
  var UID = panel.getAttribute("uid");

  // Assign this panel as the most recently opened
  inst._lastActiveChatPanelUID = UID;
  inst._lastClosedUID = null;

  // Get the chat history for this user
  if (!inst._userChatHash[UID].historyLoaded) {
    inst.getChatHistory(panel, UID);
  }

  var statusMessage = panel.getElementsByClassName("chatInfoStatus")[0];
  var statusTime = panel.getElementsByClassName("chatInfoStatusTime")[0];
  if (statusMessage &&
      statusTime &&
      inst._userChatHash[UID] &&
      inst._userChatHash[UID].status &&
      inst._userChatHash[UID].status.length > 0)
  {
    // Parse out any links so that they are clickable, each status enabled
    // service implements parseStatus(), use this to stay consistent.
    var fbService = inst._account.getService();
    var parsedStatus = fbService.parseStatus(document,
                                             inst._userChatHash[UID].status);
    parsedStatus = inst.convertLinks(parsedStatus, "origURL");
    flockBreakLongText(parsedStatus, inst.PANEL_STATUS_MAX_LENGTH);

    // Update status message
    inst.clearChildrenForNode(statusMessage);
    statusMessage.appendChild(parsedStatus);

    // Update status time
    statusTime.innerHTML = inst._userChatHash[UID].statusTimeRel;
  }

  inst.syncEventWithFacebook({"focus_chat": UID});
};

// Sync this event with facebook
gFacebookChat.chatPanelOnPopupHiding =
function fbc_chatPanelOnPopupHiding(aEvent) {
  var inst = gFacebookChat;
  inst._lastActiveChatPanelUID = null;
  if (!inst._chatIsHiding) {
    // Only send the unfocus event when the user minimizes the chat panel,
    // an unwanted unfocus is sent when switching browser tabs to facebook.com
    inst.syncEventWithFacebook({"unfocus_chat": 1});
  }
};

// Set the visibility of the chat panel, when a user is offline there is a
// message which notifies the user, set appriate verbiage for each visibility
gFacebookChat.setChatPanelVisibility =
function fbc_setChatPanelVisibility(aPanel, aVisibility) {
  var bundle = this.getBundle();

  var UID = aPanel.getAttribute("uid");
  var visibilityChangeDiv =
    aPanel.getElementsByClassName("visibilityChange")[0];
  if (visibilityChangeDiv &&
      this._userChatHash[UID] &&
      this._userChatHash[UID].name)
  {
    var visibilityText =
      bundle.formatStringFromName("facebookchat.panel.visibility."
                                    + aVisibility,
                                  [this._userChatHash[UID].name],
                                  1);
    this.setTextForNode(visibilityChangeDiv, visibilityText);
  }
  aPanel.setAttribute("status", aVisibility);
};

// Initialize the chat panel and add it to the document
gFacebookChat.createChatPanel =
function fbc_createChatPanel(aBuddy) {
  var inst = gFacebookChat;
  var UID = aBuddy.uid;

  this.getLogger().info("createChatPanel for uid: " + UID);

  var bundle = this.getBundle();
  var panelID = "panel-" + UID;

  if (this._userChatHash[aBuddy.uid]) {
    // Assign this relationship for future retrieval
    this._userChatHash[aBuddy.uid].chatPanelID = panelID;
  }

  // Helper for loading profile from avatar or name click
  var _loadProfile = function _loadProfile(aEvent) {
    gFacebookChat.loadProfileUrl(aEvent, UID);
    inst._metrics.report("FacebookChat-FriendNameClick", null);
  };

  // Create DOM nodes
  var panel = NodeBuilder.elementWithClass("panel",
                                           "chatPanel",
                                           NodeBuilder.XUL_NS);
  panel.setAttribute("id", panelID);
  panel.setAttribute("uid", UID);
  panel.setAttribute("noautohide", "true");
  panel.setAttribute("orient", "vertical");
  panel.addEventListener("click", gFacebookChat.setFrontProcess, false);
  panel.addEventListener("popupshowing",
                         gFacebookChat.chatPanelOnPopupShowing,
                         false);
  panel.addEventListener("popuphiding",
                         gFacebookChat.chatPanelOnPopupHiding,
                         false);
  panel.addEventListener("popupshown", gFacebookChat.focusTextarea, false);

  // Main wrapper
  var chat_window_wrapper =
    NodeBuilder.elementWithClass("div", "chat_window_wrapper");
  var chat_window = NodeBuilder.elementWithClass("div", "chat_window");
  var chat_header = NodeBuilder.elementWithClass("div", "chat_header");
  var header_buttons = NodeBuilder.elementWithClass("div", "header_buttons");

  // Close button
  var close = NodeBuilder.elementWithClass("a", "close");
  close.onclick = function closeWindowButtonOnclick(aEvent) {
    gFacebookChat.closeChat(aEvent, UID);
  };
  close.setAttribute("title",
    bundle.GetStringFromName("facebookchat.tab.close"));

  // Minimize button
  var minimize = NodeBuilder.elementWithClass("a", "minimize");
  minimize.onclick = function minimizeWindowButtonOnclick() {
    gFacebookChat.minimizeChat(UID);
  };
  minimize.setAttribute("title",
    bundle.GetStringFromName("facebookchat.tab.minimize"));

  header_buttons.appendChild(close);
  header_buttons.appendChild(minimize);

  var picLink = document.createElement("a");
  picLink.setAttribute("href", "#");

  // Avatar
  var avatar = NodeBuilder.elementWithClass("img", "chatInfoPic");
  avatar.setAttribute("src", this.getAvatar(aBuddy.thumbSrc));
  avatar.onclick = _loadProfile;

  // Add the avatar inside the link
  picLink.appendChild(avatar);

  // Create username header
  var chat_header_name =
    NodeBuilder.elementWithClass("div", "chat_header_name");

  var nameLink = NodeBuilder.elementWithClass("a", null);
  nameLink.onclick = _loadProfile;
  nameLink.innerHTML = aBuddy.name;

  chat_header_name.appendChild(nameLink);

  // Add all elements to the header element
  chat_header.appendChild(header_buttons);
  chat_header.appendChild(picLink);
  chat_header.appendChild(chat_header_name);
  chat_window.appendChild(chat_header);

  // Add the chat info element
  var chatInfo = NodeBuilder.elementWithClass("div", "chatInfo");
  var chatInfoStatus = NodeBuilder.elementWithClass("span", "chatInfoStatus");
  var chatInfoStatusWordBreak = NodeBuilder.elementWithClass("wbr", null);
  var chatInfoStatusTime =
    NodeBuilder.elementWithClass("span", "chatInfoStatusTime");

  chatInfo.appendChild(chatInfoStatus);
  chatInfo.appendChild(chatInfoStatusWordBreak);
  chatInfo.appendChild(chatInfoStatusTime);
  chat_window.appendChild(chatInfo);

  // Add the conversation block
  var chat_conv = NodeBuilder.elementWithClass("div", "chat_conv");
  var conv_content = NodeBuilder.elementWithClass("div", "conversationContent");
  var visibilityChange =
    NodeBuilder.elementWithClass("div", "visibilityChange");
  conv_content.appendChild(visibilityChange);
  chat_conv.appendChild(conv_content);
  chat_window.appendChild(chat_conv);

  // Add the message input
  var chat_input_div = NodeBuilder.elementWithClass("div", "chat_input_div");
  var textarea = NodeBuilder.elementWithClass("textarea", "chat_input");
  textarea.setAttribute("id", "textarea-" + UID);
  textarea.addEventListener("dragdrop",
    function textarea_onKeyup(aEvent) {
      // Make sure the content has been dropped first
      setTimeout(function textareaDrop() {
        // Trigger a change event so that the textarea resizes
        $jQ(textarea).trigger("change");
        textarea.scrollLeft = 0;
        }, 50);
    }, false);
  textarea.addEventListener("keyup",
    function textarea_onKeyup(aEvent) {
      // Remove new lines
      this.value = this.value.replace(/[\n\r]+/g, "");
    }, false);

  textarea.addEventListener("keypress",
    function textarea_onKeypress(aEvent) {
      if (aEvent.charCode) {
        // Inform chat partner that we are typing.
        if (!inst._fbttInstance) {
          // Start a "stopped typing" countdown.
          inst._fbttInstance = new FbTypingTimer(inst, aBuddy.uid);
        } else {
          // Reset the "stopped typing" countdown.
          inst._fbttInstance.reset();
        }
      }

      // If <ENTER> if pressed, send the message
      if (aEvent.keyCode == 13 && this.value.length > 0) {

        // Cancel any outstanding typing notification.
        if (inst._fbttInstance) {
          inst._fbttInstance.cancel();
        }
        // The message is sent from the current user
        inst.addMessageToPanel(inst._loggedInUserInfo,
                               aBuddy,
                               this.value,
                               new Date().getTime(),
                               "outgoing");

        // Outgoing messages are send to the current panel's UID
        inst.sendChatMessage(UID, this.value);

        // Blank the input
        this.value = "";
      }
    }, false);

  chat_input_div.appendChild(textarea);
  chat_window.appendChild(chat_input_div);

  // When the content pane is clicked, make sure to focus
  conv_content.onclick = function content_onClick(aEvent) {
    textarea.focus();
  };

  // Add the popup to the popupset
  chat_window_wrapper.appendChild(chat_window);
  panel.appendChild(chat_window_wrapper);

  // Set the online status of the user for the panel
  var availabilityStatus =
    this.getBuddyAvailabilityStatus(aBuddy.available);
  if (availabilityStatus) {
    this.setChatPanelVisibility(panel, availabilityStatus);
  }

  this.chatPopupset.appendChild(panel);

  // Auto resize the text box so that the textarea expands when the user
  // reaches the end of the current spcace
  $jQ(textarea).autoResize({ animateDuration : 300, extraSpace : 0 });
};

// Reset all tabs back to their default state
gFacebookChat.resetChatTabsState =
function fbc_resetChatTabsState() {
  // Reset the selection state of all tabs
  var tabs = this.tabBarContainer.childNodes;
  for (var i = 0; i < tabs.length; i++) {
    tabs[i].removeAttribute("selected");
  }
};

// Give the tab a visual selected state
gFacebookChat.selectTabWithState =
function fbc_selectTabWithState(aTab, aState) {
  this.resetChatTabsState();

  // Select the tab in question
  aTab.setAttribute(aState, "true");
};

// Focus the message input textarea when the panel is shown
gFacebookChat.focusTextarea =
function fbc_focusTextarea(aEvent) {
  var panel = aEvent.target;
  if (panel) {
    var textarea = panel.getElementsByTagName("textarea");
    if (textarea[0]) {
      textarea[0].focus();
      document.commandDispatcher.advanceFocus();
    }
  }
};

// Ensure the application is in the foreground/active and set the cursor in the
// input field
gFacebookChat.setFrontProcess =
function fbc_setFrontProcess(aEvent) {
  CC["@flock.com/sysinfo;1"].getService(CI.flockISystemInfo)
                            .setFrontProcess();
  gFacebookChat.focusTextarea(aEvent);
};

// When the tab is clicked either open a new panel if not already created
// of toggle it's state if already initialized
gFacebookChat.tabClick =
function fbc_tabClick(aTab, aUser, aOpenPanelOverride) {
  this._logger.info("tabClick");

  this.ensureChatPanelInit(aUser);

  var UID = aUser.uid;
  var chatPanel =
    document.getElementById(this._userChatHash[UID].chatPanelID);

  // Remove any notification badges and reset the unread count
  this.removeNotificationBadgeForUID(UID);
  this._userChatHash[UID].unreadMsgCount = 0;

  // Only show 1 chat at a time
  this.minimizeAllChats(chatPanel);

  // Make sure the tab highlight has been removed
  aTab.removeAttribute("highlight");

  // Select the current tab
  this.selectTabWithState(aTab, "selected");

  if (chatPanel.state == "closed" || aOpenPanelOverride) {
    // Open the panel
    chatPanel.openPopup(aTab, "before_end");
  } else {
    // Close the panel
    this.resetChatTabsState();
    chatPanel.hidePopup();
  }

};

/* Buddy Panel */

gFacebookChat.resizeBuddyListContainer =
function fbc_resizeBuddyListContainer() {
  var win = this.getCurrentWindow();
  if (win) {
    // We cannot use scrollHeight or clientHeight when the list is not visibile
    // calcuate the height by the number of items that the list contains
    var buddies = this.buddyList.getElementsByClassName("buddy");
    var totalBuddies = buddies.length;
    var onlineBuddies = 0;
    for (var i = 0; i < buddies.length; i++) {
      if (buddies[i].getAttribute("status") == "online") {
        onlineBuddies++;
      }
    }

    var idleOfflineBuddies = (totalBuddies - onlineBuddies);
    var buddyListHeight = (onlineBuddies * this.BUDDY_HEIGHT)
                        + (idleOfflineBuddies * this.BUDDY_HEIGHT_IDLE);

    var windowHeight = win.document.height - this.BUDDY_LIST_Y_OFFSET;
    var height;
    if (buddyListHeight > windowHeight) {
      height = windowHeight + "px";
    } else {
      height = "auto";
    }
    this.buddyList.setAttribute("style", "height: " + height + ";");
  }
};

gFacebookChat.buddyPanelOnPopupShown =
function fbc_buddyPanelOnPopupShown() {
  // Focus the search filter
  var buddyFilter = document.getElementById("buddyFilter");
  buddyFilter.focus();
};

gFacebookChat.buddyPanelOnPopupShowing =
function fbc_buddyPanelOnPopupShowing() {
  this.resizeBuddyListContainer();

  // Restart the timer interval and refresh the buddy list
  this.resetBuddyListTimer();
  this.getChatData();
};

gFacebookChat.clearBuddyFilter =
function fbc_clearBuddyFilter() {
  var buddyFilter = document.getElementById("buddyFilter");
  buddyFilter.value = "";

  // Reset the search filter
  this.keyUpBuddyFilter(null, buddyFilter);
};

gFacebookChat.focusBuddyFilter =
function fbc_focusBuddyFilter(aFilterInput) {
  var placeholderTxt = this.getBuddyFilterPlaceholder();

  if (aFilterInput.value == placeholderTxt) {
    // Blank the value if set to the palceholder text
    aFilterInput.value = "";
  }
  this.resetBuddyFilter(aFilterInput, placeholderTxt);
};

gFacebookChat.blurBuddyFilter =
function fbc_blurBuddyFilter(aFilterInput) {
  var placeholderTxt = this.getBuddyFilterPlaceholder();

  if (aFilterInput.value == "") {
    // Show the placeholder text if nothing has been entered
    aFilterInput.value = placeholderTxt;
  }
  this.resetBuddyFilter(aFilterInput, placeholderTxt);
};

gFacebookChat.keyUpBuddyFilter =
function fbc_keyUpBuddyFilter(aEvent, aFilterInput) {

  if (aEvent && aEvent.keyCode == aEvent.DOM_VK_ESCAPE) {
    // Close the buddy list panel
    this.resetGroupTabStates();
    this.buddyListPanel.hidePopup();
  }

  var placeholderTxt = this.getBuddyFilterPlaceholder();

  // Filter search results
  var buddyRegExp;
  var searchFor = aFilterInput.value;
  for (var i = 0; i < this.buddyList.childNodes.length; i++) {
    var buddy = this.buddyList.childNodes[i];
     // Case insensitive search
     buddyRegExp = new RegExp(searchFor, "i");
     var buddyInfo = this._userChatHash[buddy.getAttribute("uid")];
     if ((buddyInfo && buddyInfo.name && buddyInfo.name.match(buddyRegExp)) ||
        searchFor == "" ||
        searchFor == placeholderTxt)
     {
       buddy.removeAttribute("hide");
     } else {
       buddy.setAttribute("hide", "true");
     }
  }

  // Handle the state of the filter input
  this.resetBuddyFilter(aFilterInput, placeholderTxt);
};

gFacebookChat.resetBuddyFilter =
function fbc_resetBuddyFilter(aFilterInput, aPlaceholderTxt) {
  if (aFilterInput.value &&
      aFilterInput.value.length >= 0 &&
      aFilterInput.value != aPlaceholderTxt)
  {
    aFilterInput.parentNode.removeAttribute("placeholder");
  } else {
    aFilterInput.parentNode.setAttribute("placeholder", "true");
  }
};

// Set the current buddy count
gFacebookChat.setBuddyCount =
function fbc_setBuddyCount(aBuddyCount) {
  document.getElementById("buddyCountContainer").removeAttribute("empty");
  this.setTextForElementId("buddyCount", aBuddyCount);
};

gFacebookChat.removeBuddy =
function fbc_removeBuddy(aBuddy) {
  var buddyNode;
  this.getLogger().debug("buddy '" + aBuddy.uid
                         + "' (" + aBuddy.name + ") removed from list");
  for (var i = 0; i < this.buddyList.childNodes.length; i++) {
    buddyNode = this.buddyList.childNodes[i];
    if (buddyNode.getAttribute("uid") == aBuddy.uid) {
      // Animate removing of the buddy
      if (aBuddy.sync && aBuddy.sync == "removed") {
        $jQ(buddyNode).slideUp({duration: this.DEFAULT_ANIMATION_DURATION,
                                easing: "easeInQuad",
                                complete:
          function removeBuddy_callback() {
            if (gFacebookChat.buddyList && buddyNode) {
              gFacebookChat.buddyList.removeChild(buddyNode);
            }
          }});
      } else {
        this.buddyList.removeChild(buddyNode);
      }
      return;
    }
  }
};

gFacebookChat.addBuddy =
function fbc_addBuddy(aBuddy) {
  var inst = gFacebookChat;
  this.getLogger().debug("buddy '" + aBuddy.uid
                         + "' (" + aBuddy.name + ") added to list");
  var UID = aBuddy.uid;
  var buddy = NodeBuilder.elementWithClass("div", "buddy");
  buddy.setAttribute("uid", UID);
  buddy.onclick = function buddy_onClick(aEvent) {
    // Open tab or focus existing panel
    if (aEvent.button == 2) {
      return;
    }
    var uid = aEvent.currentTarget.getAttribute("uid");
    inst.getLogger().debug("Buddy in buddy list clicked. UID: '" + uid + "'");

    // Make sure a tab has been opened and the panel is opened
    inst.ensureTabInit(aBuddy, true);
  };

  // Set the online status of the user
  var availabilityStatus = this.getBuddyAvailabilityStatus(aBuddy.available);
  if (availabilityStatus) {
    buddy.setAttribute("status", availabilityStatus);
    if (this._userChatHash[UID]) {
      // Update the visibility of the chat tab and panel
      this.setTabVisibilityForUID(UID, availabilityStatus);

      if (this._userChatHash[UID].chatPanelID) {
        var panel =
          document.getElementById(this._userChatHash[UID].chatPanelID);
        this.setChatPanelVisibility(panel, availabilityStatus);
      }
    }
  }

  var buddyAvatar = NodeBuilder.elementWithClass("img", "buddyAvatar");
  buddyAvatar.setAttribute("src", aBuddy.thumbSrc);

  var buddyStatus = NodeBuilder.elementWithClass("div", "buddyStatus");
  var name = NodeBuilder.elementWithClass("strong", null);
  this.setTextForNode(name,
    this.truncateText(aBuddy.name, this.BUDDY_LIST_NAME_MAX_LENGTH));
  buddyStatus.appendChild(name);

  if (aBuddy.status) {
    var status = NodeBuilder.elementWithClass("span", null);
    // Truncate status text if needed
    this.setTextForNode(status, this.truncateText(aBuddy.status,
                                                  this.STATUS_MAX_LENGTH));
    buddyStatus.appendChild(status);
  }

  var availableDot = NodeBuilder.elementWithClass("div", "availableDot");
  var clearFix = NodeBuilder.elementWithClass("div", "clearFix");

  buddy.appendChild(buddyAvatar);
  buddy.appendChild(buddyStatus);
  buddy.appendChild(availableDot);
  buddy.appendChild(clearFix);

  function _animatedInsert(aSync, aBuddyNode) {
    var inst = gFacebookChat;
    if (inst._buddysLoaded && aSync && aSync == "new") {
      $jQ(aBuddyNode).highlightFade({color: "rgb(255,255,153)",
                                     speed: inst.DEFAULT_HIGHLIGHT_DURATION,
                                     complete:
                                     function highlightFade_Complete() {
                                       // The higlight effect leaves some cruft
                                       // in the styling, remove it.
                                       aBuddyNode.removeAttribute("style");
                                     }})
                     .slideDown({duration: inst.DEFAULT_ANIMATION_DURATION,
                                 easing: "easeInQuad"});
      return;
    }
    $jQ(aBuddyNode).show();
  }

  // Insert into the list in alphabetical order.
  if (this.buddyList.childNodes.length) {
    // Insert before first item in list that sorts higher.
    var buddyNode;
    var uid;
    for (var i = 0; i < this.buddyList.childNodes.length; i++) {
      buddyNode = this.buddyList.childNodes[i];
      $jQ(buddy).hide();
      // Include the online/idle state in the sort key.
      uid = buddyNode.getAttribute("uid");
      if (this._userChatHash[uid] &&
          (this._userChatHash[uid].available + this._userChatHash[uid].name) >
          (aBuddy.available + aBuddy.name))
      {
        this.buddyList.insertBefore(buddy, buddyNode);
        _animatedInsert(aBuddy.sync, buddy);
        break;
      }
    }
    // No lower-sorted buddies in list? Simply append.
    if (i === this.buddyList.childNodes.length) {
      this.buddyList.appendChild(buddy);
      _animatedInsert(aBuddy.sync, buddy);
    }
  } else {
    // This is the first buddy to be added to the list.
    this.buddyList.appendChild(buddy);
    _animatedInsert(aBuddy.sync, buddy);
  }
};

// Return a tab given a UID
gFacebookChat.getTabForUID =
function fbc_getTabForUID(aUID) {
  var tabs = this.tabBarContainer.childNodes;
  for (var i = 0; i < tabs.length; i++) {
    if (tabs[i].getAttribute("uid") == aUID) {
      return tabs[i];
    }
  }
  return null;
};

// Set the online visibility for a tab given a UID
gFacebookChat.setTabVisibilityForUID =
function fbc_setTabVisibilityForUID(aUID, aAvailabilityStatus) {
  if (this._userChatHash[aUID].tabID) {
    // Update the status of the respective tab
    var tab = document.getElementById(this._userChatHash[aUID].tabID);
    tab.setAttribute("status", aAvailabilityStatus);
  }
};

// Add a tab for a respective user
gFacebookChat.addTab =
function fbc_addTab(aBuddy) {
  var UID = aBuddy.uid;
  var bundle = this.getBundle();
  var tabID = new Date().getTime();
  this._userChatHash[UID].tabID = tabID;

  var chatTab = NodeBuilder.elementWithClass("div", "chatTab");
  chatTab.setAttribute("id", tabID);
  chatTab.setAttribute("uid", UID);
  chatTab.setAttribute("type", "chat");
  chatTab.onclick = function tab_onclick() {
    gFacebookChat.tabClick(this, aBuddy, false);
  };

  switch (aBuddy.available) {
    case "A":
      chatTab.setAttribute("status", "online");
      break;
    case "B":
      chatTab.setAttribute("status", "idle");
      break;
    case "C":
      chatTab.setAttribute("status", "offline");
      break;
  }

  // Tab name
  var tabName = NodeBuilder.elementWithClass("div", "tabName");

  // Truncate the tab name if needed
  tabName.innerHTML = this.truncateText(aBuddy.name, this.TAB_NAME_MAX_LENGTH);

  // Close button
  var tabClose = NodeBuilder.elementWithClass("div", "tabClose");
  tabClose.onclick = function closeTabButtonOnclick(aEvent) {
    gFacebookChat.closeChat(aEvent, UID);
  };
  tabClose.setAttribute("title",
                     bundle.GetStringFromName("facebookchat.tab.close"));

  chatTab.appendChild(tabName);
  chatTab.appendChild(tabClose);

  // Add the tab to the tab bar
  var tabBarContainer = this.tabBarContainer;
  if (tabBarContainer.hasChildNodes()) {
    tabBarContainer.insertBefore(chatTab, tabBarContainer.firstChild);
  } else {
    tabBarContainer.appendChild(chatTab);
  }

  // Make sure all chat tabs can still fit on the screen
  this.determineTabControllerState();

  // Make sure the added tab is in view
  this.ensureChatTabIsVisible(UID);

  this.refreshTabCount();
};

gFacebookChat.refreshTabCount =
function fbc_refreshTabCount() {
  this.tabBarContainer.setAttribute("tabCount",
                                    this.getNumberOfTotalChatTabs());
};

// Show the tooltip panel when a user is logged out
gFacebookChat.tabMouseOver =
function fbc_tabMouseOver(aEvent) {

  function _isPanelGroupMemeberOpen(aPanels) {
    for each (var panel in aPanels) {
      if (panel.state == "open") {
        return true;
      }
    }
    return false;
  }

  if (_isPanelGroupMemeberOpen(this.getPanelGroupMembers())) {
    return;
  }

  switch (aEvent.target.id) {
    case "chatStatusTab":
      this._tooltipMode = (this._svcLoggedIn) ? "chat" : "notLoggedIn";
      break;

    case "notificationsTab":
      this._tooltipMode = "notifications";
      break;

    default:
      break;
  }

  if (this.tooltipPanel.state == "closed") {
    this.tooltipPanel.setAttribute("hidden", "false");
    this.tooltipPanel.openPopup(aEvent.target, "before_end");
  }

};

// Hide the tooltip panel onmouseout
gFacebookChat.tabMouseOut =
function fbc_tabMouseOut(aEvent) {
  if (this.tooltipPanel.state == "open") {
    this.hideTooltip();
  }
};

gFacebookChat.hideTooltip =
function fbc_hideTooltip() {
  this.tooltipPanel.hidePopup();
  this.tooltipPanel.setAttribute("hidden", "true");
};

gFacebookChat.reopenActivePanels =
function fbc_reopenActivePanels() {
  if (this._lastActiveChatPanelUID) {
    var UID = this._lastActiveChatPanelUID;
    if (this._userChatHash[UID] &&
        this._userChatHash[UID].tabID)
    {
      this.ensureChatPanelInit(this._userChatHash[UID]);
      var panel = document.getElementById(this._userChatHash[UID].chatPanelID);
      var tabPosition = this.getChatTabIndexForUID(UID) * this.TAB_WIDTH;
      if (this.isTabPositionVisible(tabPosition)) {
        this.tabClick(document.getElementById(this._userChatHash[UID].tabID),
                      this._userChatHash[UID],
                      true);
      } else {
        this.resetChatTabsState();
      }
    }
  }
};

// Check to see if the user has finnished resizing the window
gFacebookChat.isDoneResizing =
function fbc_isDoneResizing(aEvent) {
  var inst = gFacebookChat;
  if ((new Date().getTime() - inst._resizeTime) >= 600) {
    inst.reopenActivePanels();

    // Clear the interval
    clearInterval(inst._resizeInterval);
    inst._resizeInterval = null;
    inst._resizeHideUID = null;
  }
};

// Window resize must determine if there is enough width to display the tabs
gFacebookChat.onResize =
function fbc_onResize(aEvent) {
  var inst = gFacebookChat;

  function _setBrowserDimensions() {
    inst._browserWidth = window.innerWidth;
    inst._browserHeight = window.innerHeight;
  }

  if (!inst._browserWidth && !inst._browserHeight) {
    // Initialize to the base dimension for comparing while resizing
    _setBrowserDimensions();
     return;
   }

  if (inst._browserWidth == window.innerWidth &&
      inst._browserHeight == window.innerHeight)
  {
    // If the browser dimensions haven't changed even though the resize
    // was fired, let's get out of here
    return;
  }

  // Update the browser dimensions since we have a change in size
  _setBrowserDimensions();

  if (!inst._chatLoggedIn) {
    // If we are logged out
    return;
  }

  inst._resizeTime = new Date().getTime();
  if (!inst._resizeInterval) {
    // When resizing the window, the panels get positioned off their anchor,
    // reopen them once the resize has completed
    inst._resizeInterval = setInterval(inst.isDoneResizing, 200);

    inst.minimizeAllChats(null);
    inst._resizeHideUID = inst._lastActiveChatPanelUID;
  }

  inst.determineTabControllerState();
};

// Make sure the scroll of the tab overflow is properly set
gFacebookChat.ensureScrollIsWithinBounds =
function fbc_ensureScrollIsWithinBounds() {
  if (!this._tabBarOverflowWidth) {
    return;
  }
  var numberChatTabs = this.getNumberOfTotalChatTabs();
  var visibleChatTabs = this.getNumberOfVisibleChatTabs(numberChatTabs);
  var maxScrollLeft = (numberChatTabs - visibleChatTabs)
                    * this.TAB_WIDTH;
  if (this.tabBarOverflow.scrollLeft > maxScrollLeft) {
    this.tabBarOverflow.scrollLeft = maxScrollLeft;
  }
};

gFacebookChat.getChatTabIndexForUID =
function fbc_getChatTabIndexForUID(aUID) {
  var tabs = this.tabBarContainer.childNodes;
  var tabIndex = 0;
  for (var i = 0; i < tabs.length; i++) {
    if (tabs[i].getAttribute("uid") == aUID) {
      return i;
    }
  }
  return null;
};

gFacebookChat.isTabPositionVisible =
function fbc_isTabPositionVisible(aTabPosition) {
  var overflowPosition = this._tabBarOverflowWidth
                       + this.tabBarOverflow.scrollLeft;
  if (!this._tabBarOverflowWidth ||
      (aTabPosition >= this.tabBarOverflow.scrollLeft &&
       aTabPosition < overflowPosition))
  {
    return true;
  }
  return false;
};

// Make sure the respective chat tab is in view
gFacebookChat.ensureChatTabIsVisible =
function fbc_ensureChatTabIsVisible(aUID) {
  if (!this._tabBarOverflowWidth) {
    return;
  }

  var tabPosition = this.getChatTabIndexForUID(aUID) * this.TAB_WIDTH;

  if (!this.isTabPositionVisible(tabPosition)) {
    // The tab is hidden to the right
    var totalChatTabs = this.getNumberOfTotalChatTabs();
    var visibleChatTabs = this.getNumberOfVisibleChatTabs(totalChatTabs);
    var scrollPosition = tabPosition
                       - ((visibleChatTabs - 1) * this.TAB_WIDTH);

    this.tabBarOverflow.scrollLeft = scrollPosition;
    this.setLeftRightScrollTabValues();
  }
};

// Return the number of currently initialized chat tabs
gFacebookChat.getNumberOfTotalChatTabs =
function fbc_getNumberOfTotalChatTabs() {
  return this.tabBarContainer.getElementsByClassName("chatTab").length;
};

// Return the number of currently visibile chat tabs
gFacebookChat.getNumberOfVisibleChatTabs =
function fbc_getNumberOfVisibleChatTabs(aTotalNumberChatTabs) {
  // If the overflow width isn't set, this means all tabs are visible
  return (this._tabBarOverflowWidth)
        ? this._tabBarOverflowWidth / this.TAB_WIDTH
        : aTotalNumberChatTabs;
};

// Determine the number of tabs allowed on screen without breaking the layout
gFacebookChat.getAllowedNumberOfTabs =
function fbc_getAllowedNumberOfTabs(aNumberTabs) {
  var statusBar = document.getElementById("status-bar");
  if (statusBar.boxObject) {
    var statusBarWidth = statusBar.boxObject.width;
    while ((aNumberTabs * this.TAB_WIDTH) >
            (statusBarWidth - this.STATUS_BAR_OFFSET))
    {
      aNumberTabs--;
    }
  }
  return aNumberTabs;
};

gFacebookChat.determineTabControllerState =
function fbc_determineTabControllerState() {
  var numberChatTabs = this.getNumberOfTotalChatTabs();
  var allowedNumberOfTabs = this.getAllowedNumberOfTabs(numberChatTabs);

  if (allowedNumberOfTabs != numberChatTabs) {
    this.setTabControllerState(true);

    this._tabBarOverflowWidth = this.TAB_WIDTH * allowedNumberOfTabs;

    // Only show the tabs within the explicit width
    var overflowCSS = "width: " + this._tabBarOverflowWidth + "px;"
                    + "max-width: " + this._tabBarOverflowWidth + "px;";
    this.tabBarOverflow.setAttribute("style", overflowCSS);

    // Make sure we have a proper scroll region
    this.ensureScrollIsWithinBounds();

    // Initialize the state of the left and right scroll arrows
    this.setLeftRightScrollTabValues();
  } else {
    this.setTabControllerState(false);

    // Reset the style of the overflow container
    this._tabBarOverflowWidth = null;
    this.tabBarOverflow.removeAttribute("style");
  }
};

gFacebookChat.setTabControllerState =
function fbc_setTabControllerState(aState) {
  if (aState) {
    this.tabBarOverflow.setAttribute("tabOverflow", "true");
  } else {
    this.tabBarOverflow.removeAttribute("tabOverflow");
  }
  this.chatStatusBarPanel.setAttribute("tabController", aState);
};

// Animate the scroll
gFacebookChat.animateScrollLeft =
function fbc_animateScrollLeft(aElement, aScrollLeft) {
  var inst = gFacebookChat;
  if (aElement) {
    inst._tabBarIsAnimating = true;
    inst.resetAllChats();

    // Setup the callback
    var animationCallback = function animateScrollLeft_Callback() {
      inst.setLeftRightScrollTabValues();

      // Determine if a panel was open before the scroll occured, if so
      // open the panel if the tab is still visible
      inst.reopenActivePanels();

      inst._tabBarIsAnimating = false;
    };

    // Animate the scroll
    $jQ(aElement).animate({scrollLeft: aScrollLeft},
                          220,
                          "",
                          animationCallback);
  }
};

// Sroll the tabs to the left
gFacebookChat.tabScrollLeft =
function fbc_tabScrollLeft(aEvent) {
  var target = document.getElementById("scrollLeftTab");
  if (target.hasAttribute("disabled") || this._tabBarIsAnimating) {
    return;
  }

  this.animateScrollLeft(this.tabBarOverflow,
                         (this.tabBarOverflow.scrollLeft - this.TAB_WIDTH));
};

// Sroll the tabs to the right
gFacebookChat.tabScrollRight =
function fbc_tabScrollRight(aEvent) {
  var target = document.getElementById("scrollRightTab");
  if (target.hasAttribute("disabled") || this._tabBarIsAnimating) {
    return;
  }
  this.animateScrollLeft(this.tabBarOverflow,
                         (this.tabBarOverflow.scrollLeft + this.TAB_WIDTH));
};

gFacebookChat.setLeftRightScrollTabValues =
function fbc_setLeftRightScrollTabValues() {
  var scrollLeftTab = document.getElementById("scrollLeftTab");
  var scrollRightTab = document.getElementById("scrollRightTab");

  var scrollLeftCount = document.getElementById("scrollLeftCount");
  var scrollRightCount = document.getElementById("scrollRightCount");

  // Determine the index of the left most visible tab
  var tabIndex = this.tabBarOverflow.scrollLeft / this.TAB_WIDTH;

  // Determine the number of tabs
  var totalChatTabs = this.getNumberOfTotalChatTabs();
  var visibleChatTabs = this.getNumberOfVisibleChatTabs(totalChatTabs);

  // Give the proper state to the left scroll tab
  if (tabIndex == 0) {
    this.setTextForNode(scrollLeftCount, "0");
    scrollLeftTab.setAttribute("disabled", "true");
  } else {
    this.setTextForNode(scrollLeftCount, tabIndex);
    scrollLeftTab.removeAttribute("disabled");
  }

  // Give the proper state to the right scroll tab
  if (tabIndex == (totalChatTabs - visibleChatTabs)) {
    this.setTextForNode(scrollRightCount, "0");
    scrollRightTab.setAttribute("disabled", "true");
  } else {
    this.setTextForNode(scrollRightCount,
                        (totalChatTabs - visibleChatTabs - tabIndex));
    scrollRightTab.removeAttribute("disabled");
  }

};

/* The Group Tabs/Panels represent those that can only have 1 open at a time
 * either the Chat Status Panel, Notification Panel, or Buddy List Panel
 */
gFacebookChat.resetGroupTabStates =
function fbc_resetGroupTabStates() {
  var tabs = this.getTabGroupMembers();
  for each (var tab in tabs) {
    tab.removeAttribute("selected");
  }
};

gFacebookChat.resetGroupPanelStates =
function fbc_resetGroupPanelStates() {
  var panels = this.getPanelGroupMembers();
  for each (var panel in panels) {
    if (panel.state == "open") {
      panel.hidePopup();
    }
  }
};

// Toggle the state of the buddy list and chat status panels
gFacebookChat.togglePanel =
function fbc_togglePanel(aEvent, aPanelId) {
  if (!this._svcLoggedIn) {
    // Redirect to facebook login page
    if (aEvent.button == 2) {
      return;
    }

    // Double check for a authenticated facebook account since the user is
    // attempting to login, login if there is one found
    var authenticatedAccountExists = this.checkForAuthenticatedAccount();
    if (authenticatedAccountExists) {
      return;
    }

    var win = this.getCurrentWindow();
    if (win) {
      var loginURL = this.getWD().getLocalizedString("facebookchat",
                                                     "loginURL",
                                                      null);
      win.FlockTabs.openURL(loginURL, aEvent);
    }
    return;
  }

  function _handlePanelToggle(aPanel, aTab, aPosition, aX, aY) {
    switch (aPanel.state) {
      case "open":
        aPanel.hidePopup();
        break;

      case "closed":
        gFacebookChat.resetGroupPanelStates();
        aTab.setAttribute("selected", "true");
        aPanel.openPopup(aTab, aPosition, aX, aY);
        break;
    }
  }

  this.resetGroupTabStates();

  switch (aPanelId) {
    case "buddyListPanel":
      _handlePanelToggle(this.buddyListPanel, this.buddyListTab,
                         "before_start", 0, 0);
      break;

    case "chatStatusPanel":
      _handlePanelToggle(this.chatStatusPanel, this.chatStatusTab,
                         "before_end", 0, 0);
      break;

    case "notificationsPanel":
      _handlePanelToggle(this.notificationsPanel, this.notificationsTab,
                         "before_end", 30, 0);
      // Mark the notifications read when viewing them
      this.markNotificationsRead();
      break;

    default:
      break;
  }

};

/* Chat Status Panel */

gFacebookChat.chatStatusOnPopupShowing =
function fbc_chatStatusxOnPopupShowing(aEvent) {
  if (!this._svcLoggedIn) {
    // Not logged into facebook so don't show the panel
    aEvent.preventDefault();
  } else {
    // Refresh the panel data before opening
    this.refreshAccountInfo();
  }
};

gFacebookChat.refreshVisibility =
function fbc_refreshVisibility() {
  if (!this._chatLoggedIn) {
    // Minimize all chat panels when status is changed to offline
    this.resetAllChats();
    this.chatLogout();
  } else {
    this.chatLogin();
  }

  // Show the new visibility state
  this.setChatStatus(this.getChatStatus());
};

gFacebookChat.toggleVisibility =
function fbc_toggleVisibility() {
  var newVisibility = !this._chatLoggedIn;
  var inst = this;
  var toggleVisibilityListener = {
    onSuccess: function toggleVisibility_onSuccess(aSubject, aTopic) {
      inst._logger.debug("Setting visibility to " + newVisibility);
      inst._chatLoggedIn = newVisibility;
      inst.refreshVisibility();

      var visibilityJS = inst.getWD().getLocalizedString("facebookchat",
                                                         "visibilityJS",
                                                         null)
                                     .replace("%visibility%", newVisibility);

      // Check each browser instance for active Facebook tabs
      var browser;
      var tabWindow;
      var wm = CC["@mozilla.org/appshell/window-mediator;1"]
               .getService(CI.nsIWindowMediator);
      var browserEnumerator = wm.getEnumerator("navigator:browser");
      while (browserEnumerator.hasMoreElements()) {
        browser = browserEnumerator.getNext().getBrowser();
        // Get all facebook windows and change visibility on each tab
        for (var i = 0; i < browser.mTabs.length; i++) {
          if (inst.isURIFacebook(browser.mTabs[i].linkedBrowser.currentURI)) {
            tabWindow = browser.mTabs[i].linkedBrowser.docShell
                               .QueryInterface(CI.nsIInterfaceRequestor)
                               .getInterface(CI.nsIDOMWindow);
            // Update the status on the facebook tab
            try {
              eval("tabWindow.wrappedJSObject." + visibilityJS);
              return;
            } catch (ex) {
              // Unable to update status
            }
          }
        }
      }
    },
    onError: function toggleVisibility_onError(aFlockError, aTopic) {
    }
  };

  // Show a loading state
  this.setChatStatus("loading");

  // Set the new visibility
  this.setVisibility(toggleVisibilityListener, newVisibility);
};

// Open the Facebook Chat help page in a new tab
gFacebookChat.openChatHelpURL =
function fbc_openChatHelpURL(aEvent) {
  this.getLogger().info("openChatHelpURL");

  if (aEvent.button == 2) {
    // Right click
    return;
  }

  var chatHelpURL = this.getWD().getLocalizedString("facebookchat",
                                                    "chatHelpURL",
                                                    null);
  // Open the URL in a new tab
  this.openURL(chatHelpURL);

  // Hide the panel
  this.chatStatusPanel.hidePopup();
};

// Return the user avatar, if set, otherwise return the blank avatar
gFacebookChat.getAvatar =
function fbc_getAvatar(aAvatar) {
  if (aAvatar && aAvatar.length > 0) {
    return aAvatar;
  } else {
    return this.getWD().getLocalizedString("facebookchat","blankAvatar", null);
  }
};

// Set the status text in both the input and text field, if none is provided
// get the respective information from the service
gFacebookChat.setChatPanelStatus =
function fbc_setChatPanelStatus(aStatus) {
  // Update the time of status last updated
  var statusLastUpdated = this._account.getParam("lastStatusMessageUpdateDate");
  if (!statusLastUpdated) {
    statusLastUpdated = new Date();
  }
  var friendlyDate =
    FlockDateFormatter.getFriendlyLastDate(new Date(statusLastUpdated * 1000));
  this.setTextForElementId("statusUpdateTime", friendlyDate);

  // Set the status
  var status = (aStatus) ? aStatus : this._account.getEditableStatus();
  this.setTextForElementId("statusText", status);
  document.getElementById("statusInputText").value = status;
  return status;
};

gFacebookChat.getStatusMode =
function fbc_getStatusMode() {
  return document.getElementById("statusUpdates").getAttribute("mode");
};

gFacebookChat.setStatusMode =
function fbc_setStatusMode(aStatusMode) {
  if (aStatusMode) {
    document.getElementById("statusUpdates")
            .setAttribute("mode", aStatusMode);
  }
};

gFacebookChat.resetStatusState =
function fbc_resetStatusState(aStatus) {
  // Update the status on screen
  var status = this.setChatPanelStatus(aStatus);

  if (status && status.length > 0) {
    this.setStatusMode("status");
  } else {
    // Display no status message
    this.setStatusMode("empty");
  }
};

gFacebookChat.toggleEditStatus =
function fbc_toggleEditStatus() {
  var mode = this.getStatusMode();
  if (mode && mode == "edit") {
    this.resetStatusState(null);
  } else {
    this.setStatusMode("edit");

    // Focus the status text input box
    document.getElementById("statusInputText").focus();
  }
};

gFacebookChat.ensureAccountInfoInit =
function fbc_ensureAccountInfoInit() {
  if (!this._loggedInUserInfo) {
    this.refreshAccountInfo();
  }
};

gFacebookChat.refreshAccountInfo =
function fbc_refreshAccountInfo() {
  // Set the current status
  var status = this._account.getEditableStatus();
  this.resetStatusState(status);

  var name;
  if (this._matchedName) {
    // We have a matched name from the document
    name = this._matchedName;
  } else {
    // If no matched name, take the name from the account
    name = this._account.getParam("name");
    if (name.indexOf(" ") != -1) {
      // Take only the first name
      name = name.split(" ")[0];
    }
  }
  this.setTextForElementId("nameText", name);
  this.setTextForElementId("nameInputText", name);

  // Set the user avatar
  var avatarURL = this.getAvatar(this._account.getParam("avatar"));
  document.getElementById("chatPanelUserAvatar")
          .setAttribute("src", avatarURL);

  // Create object of the current logged in user's information
  this._loggedInUserInfo = {
    uid: this._UID,
    status: status,
    thumbSrc: avatarURL,
    name: name
  };

  var history = {};
  if (this._userChatHash[this._UID] && this._userChatHash[this._UID].history) {
    // If an existing chat history has been set, use it when refreshing the
    // logged in user's account info
    history = this._userChatHash[this._UID].history;
  }
  this._userChatHash[this._UID] = this._loggedInUserInfo;
  this._userChatHash[this._UID].history = history;
};

// OnKeyUp handler for the status input within the chat status panel
gFacebookChat.statusTextKeyup =
function fbc_statusTextKeyup(aTextInput, aEvent) {
  var statusMessage = aTextInput.value;
  // Remove new lines
  statusMessage.replace(/[\n\r\t]/g, "");

  switch (aEvent.keyCode) {
    case aEvent.DOM_VK_ESCAPE:
      // Exit from edit mode
      this.resetStatusState(null);
      break;

    case aEvent.DOM_VK_RETURN:
      // Set the status
      this.setAccountStatus(statusMessage);
      aTextInput.value = "";
      break;

    default:
      break;
  }
};

// Set the user's status message
gFacebookChat.setAccountStatus =
function fbc_setAccountStatus(aStatusMessage) {
  if (!this._svcLoggedIn) {
    return;
  }
  var logger = this.getLogger();

  // Update displayed status message
  this.setStatusMode("saving");

  // Only show loading div
  this.setChatStatus("loading");

  var inst = this;
  var setAccountStatusListener = {
    onSuccess: function setStatusListener_onSuccess(aSubject, aTopic) {
      logger.debug("setStatusListener_onSuccess: " + aSubject);
      // Go back to normal status mode
      gFacebookChat.resetStatusState(aStatusMessage);
      inst.setChatStatus("online");
    },

    onError: function setStatusListener_onError(aFlockError, aTopic) {
      logger.error("setStatusListener_onError: " + aFlockError);
      inst.setChatStatus("online");
    }

  };
  this._account.setStatus(aStatusMessage, setAccountStatusListener);
  this._metrics.report("FacebookChat-StatusChange", null);
};

gFacebookChat.getChatStatus =
function fbc_getChatStatus() {
  if (this._svcLoggedIn) {
    return (this._chatLoggedIn) ? "online" : "offline";
  }
  return "error";
};

// Set the user's chat status, either hidden, online, offline, or idle
gFacebookChat.setChatStatus =
function fbc_setChatStatus(aStatus) {
  if (!this._chatVisible && aStatus != "hidden") {
    // Don't set the status when the chat isn't visible unless we are making
    // an attempt to hide it
    return;
  }

  if (aStatus) {
    // Keep the last state visibile during a loading state change
    // via CSS selectors
    this.chatStatusPanel.setAttribute("lastStatus",
      this.chatStatusPanel.getAttribute("status"));

    this.chatStatusPanel.setAttribute("status", aStatus);
    this.chatStatusBarPanel.setAttribute("status", aStatus);
  }
};

/* Notification Badges */

gFacebookChat.getBadgePosition =
function fbc_getBadgePosition(aType) {
  switch (aType) {
    case "chat":
      return { x: this.BADGE_CHAT_X,
               y: this.BADGE_CHAT_Y };
    case "notification":
      return { x: this.BADGE_NOTIFICATION_X,
               y: this.BADGE_NOTIFICATION_Y };
    default:
      return { x: 0, y: 0};
  }
};

gFacebookChat.getBadgeForTab =
function fbc_getBadgeForTab(aTab) {
  switch (aTab.getAttribute("type")) {
    case "chat":
      var UID = aTab.getAttribute("uid") ? aTab.getAttribute("uid") : null;
      if (this._userChatHash[UID] && this._userChatHash[UID].badgeID) {
        return document.getElementById(this._userChatHash[UID].badgeID);
      }
      break;

    case "notification":
      return document.getElementById(this._notificationBadgeID);
  }
  return null;
};

gFacebookChat.animateBadge =
function fbc_animateBadge(aBadge) {
  if (aBadge) {
    var badgeDiv = aBadge.getElementsByTagName("div")[1];
    $jQ(badgeDiv).effect("bounce", { times:3 }, 300);
  }
};

gFacebookChat.removeNotificationBadgeForUID =
function fbc_removeNotificationBadgeForUID(aUID) {
  if (this._userChatHash[aUID].badgeID) {
    var notificationBadge =
      document.getElementById(this._userChatHash[aUID].badgeID);
    notificationBadge.parentNode.removeChild(notificationBadge);
    this._userChatHash[aUID].badgeID = null;
  }
};

// Remove the badge for the notifications tab
gFacebookChat.removeNotificationTabBadge =
function fbc_removeNotificationTabBadge() {
  if (this._notificationBadgeID) {
    var notificationBadge =
      document.getElementById(this._notificationBadgeID);
    if (notificationBadge) {
      notificationBadge.parentNode.removeChild(notificationBadge);
      this._notificationBadgeID = null;
    }
  }
};

gFacebookChat.ensureNotificationBadgeInit =
function fbc_ensureNotificationBadgeInit(aTab, aCount) {
  var notificationBadge = this.getBadgeForTab(aTab);
  if (!notificationBadge) {
    // Keep a reference of the currently focused element, so we can return
    // focus to it once the badge has been added
    var focusedElement = document.commandDispatcher.focusedElement;
    // Create the badge
    this.addNotificationBadge(aTab, aCount);
    if (focusedElement) {
      // Return focus
      focusedElement.focus();
    }
  } else {
    // Update the count
    var spanCount = notificationBadge.getElementsByTagName("span")[0];
    spanCount.innerHTML = aCount;
    this.animateBadge(notificationBadge);
  }
};

gFacebookChat.addNotificationBadge =
function fbc_addNotificationBadge(aTab, aCount) {
  var inst = gFacebookChat;

  // The badge is either meant for a chat tab, or the notifications tab itself
  var UID = aTab.getAttribute("uid") ? aTab.getAttribute("uid") : null;
  var type = aTab.getAttribute("type");
  var badgeID = (UID) ? ("badge-" + UID) : "badge-notificationsTab";

  // Create the DOM nodes
  var panel = NodeBuilder.elementWithClass("panel",
                                           "notificationBadgePanel",
                                           NodeBuilder.XUL_NS);
  panel.setAttribute("id", badgeID);
  panel.setAttribute("noautohide", "true");
  panel.setAttribute("orient", "vertical");
  panel.setAttribute("type", type);
  if (UID) {
    // If we are dealing with a chat tab, associate it with the respective UID
    this._userChatHash[UID].badgeID = badgeID;
    panel.setAttribute("uid", UID);
    panel.addEventListener("popupshown",
                           function badge_popupShown(aEvent) {
                             var badge = gFacebookChat.getBadgeForTab(aTab);
                             gFacebookChat.animateBadge(badge);
                             this.setAttribute("popupshown", null);
                           },
                           false);
  }
  var div = NodeBuilder.elementWithClass("div", "badgeContainer");
  var badge = NodeBuilder.elementWithClass("div", "notificationBadge");
  var count = NodeBuilder.elementWithClass("span", "notificationBadgeCount");
  count.innerHTML = (aCount) ? aCount : 0;
  div.appendChild(badge);
  badge.appendChild(count);
  panel.appendChild(div);
  this.badgePopupset.appendChild(panel);

  if (this._chatVisible && this.isStatusBarVisible()) {
    // Only open when the chat is visible
    var position = this.getBadgePosition(type);
    panel.openPopup(aTab, "before_end", position.x, position.y);
  }

  var panelOnClickFunc = null; 
  switch (type) {
    case "chat":
      panelOnClickFunc = function chatPanelOnClick(aEvent) {
        inst.tabClick(document.getElementById(inst._userChatHash[UID].tabID),
                      inst._userChatHash[UID],
                      true);
      };
      break;

    case "notification":
      panelOnClickFunc = function notificationPanelOnClick(aEvent) {
        inst.togglePanel(aEvent, "notificationsPanel");
      };
      this._notificationBadgeID = badgeID;
      break;
  }

  // Assign the event listener to handle clicks
  panel.addEventListener("click", panelOnClickFunc, false);
};

/* Utility Functions */

gFacebookChat.isStatusBarVisible =
function fbc_isStatusBarVisible() {
  var win = this.getCurrentWindow();
  var chromeHidden = win.document.getElementById("main-window")
                                 .getAttribute("chromehidden");
  return (chromeHidden.indexOf("status") == -1);
};

gFacebookChat.getBuddyAvailabilityStatus =
function fbc_getBuddyAvailabilityStatus(aAvailable) {
  switch (aAvailable) {
    case "A":
      return "online";
    case "B":
      return "idle";
    case "C":
      return "offline";
  }
  return null;
};

// Given two UIDs, return the UID of the logged in user's buddy
gFacebookChat.getBuddyUID =
function fbc_getBuddyUID(aUID1, aUID2) {
  var buddyUID = (aUID1 == this._UID) ? aUID2 : aUID1;
  if (buddyUID != this._UID) {
    // Double check that this is not the logged in user
    return buddyUID;
  }
  return null;
};

gFacebookChat.clearChildrenForNode =
function fbc_clearChildrenForNode(aNode) {
  while (aNode.childNodes.length > 0) {
    aNode.removeChild(aNode.lastChild);
  }
};

// Links that are tied to the document need to be within browser scope
// convert them so the links will work properly
gFacebookChat.convertLinks =
function fbc_convertLinks(aNode, aLinkAttribute) {
  var links = aNode.getElementsByTagName("a");
  for (var i = 0; i < links.length; i++) {
    var link = links[i];
    var href = link.getAttribute(aLinkAttribute);
    if (href && href.length > 0) {
      link.onmouseup = function _linkOnClick(aEvent) {
        gFacebookChat.openURL(href);
      };
    }
    link.onmouseover = null;
    link.onmouseout = null;
    link.removeAttribute("target");
    link.removeAttribute("href");
  }
  return aNode;
};

gFacebookChat.openURL =
function fbc_openURL(aURL) {
  var win = this.getCurrentWindow();
  if (win && aURL) {
    // Open the URL in a new tab
    win.openUILinkIn(aURL, "tab");
  }
};

// Helper function to format the time properly
gFacebookChat.formatMessageDate =
function fbc_formatMessageDate(aDate) {
  var suffix = "";
  var currentHour = aDate.getHours();
  if (currentHour < 12) {
    suffix = "am";
  } else {
    suffix = "pm";
  }
  if (currentHour == 0) {
     currentHour = 12;
  } else if (currentHour > 12) {
     currentHour = currentHour - 12;
  }
  // Zero pad minutes
  var currentMinute = aDate.getMinutes() + "";
  if (currentMinute.length == 1) {
    currentMinute = "0" + currentMinute;
  }
  return currentHour + ":" + currentMinute + suffix;
};

// Set the text given a respective element id
gFacebookChat.setTextForElementId =
function fbc_setTextForElementId(aDivId, aText) {
  var node = document.getElementById(aDivId);
  // Clean up existing node
  while (node.childNodes.length > 0) {
    node.removeChild(node.lastChild);
  }
  node.appendChild(document.createTextNode(aText));
};

gFacebookChat.setTextForNode =
function fbc_setTextForNode(aNode, aText) {
  // Clean up existing node
  while (aNode.childNodes.length > 0) {
    aNode.removeChild(aNode.lastChild);
  }
  aNode.appendChild(document.createTextNode(aText));
};

// Truncate a string longer than the respective max length
gFacebookChat.truncateText =
function fbc_truncateText(aText, aMaxLength) {
  if (aText && aText.length > aMaxLength) {
    aText = aText.substring(0, aMaxLength)
          + "...";
  }
  return aText;
};

/* Tooltip Panel */

// Display the proper tooltip text
gFacebookChat.tooltipPanelOnPopupShowing =
function fbc_tooltipPanelOnPopupShowing(aEvent) {
  var propertiesString = null;

  switch (this._tooltipMode) {
    case "chat":
      propertiesString = "facebookchat.chatPanel.title";
      break;

    case "notLoggedIn":
      propertiesString = "facebookchat.chatPanel.notLoggedIn";
      break;

    case "notifications":
      propertiesString = "facebookchat.notificationsPanel.title";
      break;
  }

  if (propertiesString) {
    this.setTextForElementId("tooltipPanelText",
      this.getBundle().GetStringFromName(propertiesString));
  }

};

/******************************************************************************
 * Initialization
 ******************************************************************************/

gFacebookChat.init =
function fbc_init() {
  if (gFacebookChat._isInitialized) {
    return;
  }
  var inst = gFacebookChat;
  inst._isInitialized = true;

  // Ensure the logger is initialized.
  inst.getLogger().info("init");

  // Determine whether this feature is enabled
  var prefs = inst.getPrefs();
  if (prefs.getPrefType(inst.PREF_CHAT_ENABLED) &&
      !prefs.getBoolPref(inst.PREF_CHAT_ENABLED))
  {
    inst._isEnabled = false;
    inst.setChatStatus("hidden");
    inst._logger.debug("Feature not enabled, skipping init");
    return;
  }

  inst.getWD();
  inst._Api = new FacebookChatApi(inst._logger, inst._wd);

  // Register to listen for the "flock-process-state" event on Mac
  inst._obsService = CC["@mozilla.org/observer-service;1"]
                     .getService(CI.nsIObserverService);
  if (CC["@mozilla.org/xre/app-info;1"]
        .getService(CI.nsIXULAppInfo)
        .QueryInterface(CI.nsIXULRuntime).OS == "Darwin")
  {
    inst._obsService.addObserver(inst, "flock-process-state", false);
    inst._processStateObserverAdded = true;
  }

  // Initiliaze the HTTP header parser
  inst._headerParser = new FbHeaderParser();

  // Initialize the emotions parser
  inst._emotionsParser = new FbEmotionsParser();

  // Inject two popupset containers, one for all chat panels and the other
  // for notification badges (new messages or new notifications)
  inst.injectPopupsets();

  // Add the Accounts Listener for auth changes
  inst.addAccountsObserver();

  // Check for a previously authenticated account
  gFacebookChat.checkForAuthenticatedAccount();

  // Add resize event listener
  window.addEventListener("resize", inst.onResize, false);

  // Add Location change listener
  getBrowser().addProgressListener(inst, CI.nsIWebProgress.NOTIFY_LOCATION);

  // Keep a handle to the Metrics service
  inst._metrics = CC["@flock.com/metrics-service;1"]
                  .getService(CI.flockIMetricsService);
  inst.sentMessageCount = 0;
};

gFacebookChat.observe =
function fbc_observe(aSubject, aTopic, aData) {
  if (aTopic == "flock-process-state") {
    if (aData == "foreground") {
      this.resetChatTabsState();
    }
    if (aData == "background") {
      this.minimizeAllChats(null);
    }
  }
};

gFacebookChat.checkForAuthenticatedAccount =
function fbc_checkForAuthenticatedAccount() {
  this._logger.debug("checkForAuthenticatedAccount");
  var facebookService = this.getFacebookSvc();
  var accountUrn = facebookService.getAuthenticatedAccount();
  if (accountUrn) {
    var account = facebookService.getAccount(accountUrn);
    // We already have an authenticated account
    if (account) {
      this.getLogger().info("checkForAuthenticatedAccount: found account: "
                            + accountUrn);
      this.svcLogin(account);
      return true;
    }
  }
  return false;
};

gFacebookChat.injectPopupsets =
function fbc_injectPopupsets() {
  var mainWindow = document.getElementById("main-window");

  // Popupset for all chat panels
  var chatPopupset =
    NodeBuilder.elementWithClass("popupset", null, NodeBuilder.XUL_NS);
  chatPopupset.setAttribute("id", "chatPopupset");

  // Popupset for all notification badges
  var notificationBadgePopupset =
    NodeBuilder.elementWithClass("popupset", null, NodeBuilder.XUL_NS);
  notificationBadgePopupset.setAttribute("id", "badgePopupset");

  mainWindow.appendChild(chatPopupset);
  mainWindow.appendChild(notificationBadgePopupset);
};

gFacebookChat.uninit =
function fbc_uninit() {
  var inst = gFacebookChat;
  if (!inst._isInitialized) {
    return;
  }
  inst._isInitialized = false;
  inst._logger.info("uninit");

  // Do nothing else if the feature was not enabled
  if (!inst._isEnabled) {
    inst._logger.debug("Feature not enabled, skipping uninit");
    return;
  }

  // Unload the header parser
  inst._headerParser.uninit();

  // Stop the incoming chat stream and cancel the timer
  inst.chatLogout();

  // Remove resize event listener
  window.removeEventListener("resize", inst.onResize, false);

  // Remove location change listener
  getBrowser().removeProgressListener(inst, CI.nsIWebProgress.NOTIFY_LOCATION);

  // Remove observers
  if (inst._accountsObserver) {
    inst.getAccountsSvc().removeObserver(inst._accountsObserver);
  }
  if (inst._processStateObserverAdded) {
    inst._obsService.removeObserver(inst, "flock-process-state", false);
  }
};

// Initialize
window.addEventListener("load", gFacebookChat.init, false);
window.addEventListener("unload", gFacebookChat.uninit, false);
