/*
  Frolg -- Z-machine assembler
  version 1.2
  Licensed under GNU GPL v3 or later version
*/

#define VERSION "1.2"

#ifdef _WIN32
#include <fcntl.h>
#define strncasecmp strnicmp
#else
#include <strings.h>
#endif
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

typedef unsigned char byte;

typedef struct Macro {
  char*name;
  char kind;
  union {
    // 0=undefined
    char*text; // 1=macro
    int data; // 2=numeric
    // 3=IF
    // 4=ELSE
    // 5=ENDIF
    char cmd; // 6=primitive
    // 7=ENDMAC
    // 8=instruction (data: bit8=extended, bit9=text/varbranch)
    // 9=primitive requiring a name (cmd)
    // 10=MRETURN
    // 11=primitive instruction
    // 12=block start
    // 13=block end
  };
  struct Macro*next;
} Macro;

typedef struct Object {
  char*name;
  byte attr[6];
  int loc;
  int next;
  int first;
  int table; // If the property table isn't normal, this is its address
  byte size[128];
  byte*data[128];
} Object;

const char alphabet_table[]="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ ^0123456789.,!?_#'\"/\\-:()";
const char alphabet_table_old[]="abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ 0123456789.,!?_#'\"/\\<-:()";

const byte cost_v1[3*3*3]={1,2,2,2,2,3,2,3,2,2,2,3,2,1,2,3,2,2,2,3,2,3,2,2,2,2,1};
const byte cost_v3[3*3*3]={1,3,3,2,4,4,2,4,3,2,3,4,2,1,4,3,4,3,2,4,3,3,5,4,2,4,1};

const char state_v1[3*3*3*6]=
  "*-----" "*4----" "*5----"
  "5*----" "3*----" "3*4---"
  "4*----" "2*5---" "2*----"

  "2*----" "4*----" "2*5---"
  "*5----" "*-----" "*4----"
  "3*4---" "5*----" "3*----"

  "3*----" "3*4---" "5*----"
  "2*5---" "2*----" "4*----"
  "*4----" "*5----" "*-----"
;

const char state_v3[3*3*3*6]=
  "*-----" "*44---" "*55---"
  "5*----" "5*44--" "5*55--"
  "4*----" "4*44--" "*-----" // It automatically consider "6??" in this case

  "4*----" "44*---" "4*55--"
  "*5----" "*-----" "*555--"
  "44*---" "444*--" "*-----"

  "5*----" "5*44--" "55*---"
  "55*---" "55*44-" "555*--"
  "*4----" "*444--" "*-----"
;

byte mem[0x10000];
int here;
Macro*macro;
int stack[8192];
byte stacktype[8192];
int stackptr;
int stackbase;
int pass;
Macro*label;
int defining;
int skipping;
int synonyms;
char*syntext[96];
int dynfwords;
int mac_args;
Macro*label1;
int block_count;
int block_addr[8192];
int block_stack[256];
int block_ptr;
Macro label_clip;
Macro*long_text_macro;
char long_text_data[8192];
char*long_text_ptr;
int compliant;
int maxpass=6;
int macloop;
int obj_count=1;
Object*obj_data;
int property_defaults[64];
int min_property_defaults=64;
FILE*filin;
FILE*filout;
char*story_name;
byte encoding=0;
//byte uuid[16];

#define Push(x) (stack[stackptr++ +stackbase]=(x))
#define Pop (stacktype[stackbase+stackptr-1]=0,stack[stackbase+--stackptr])
#define Type(x) (stacktype[stackbase+stackptr]|=(x))
#define Arg(x) (stack[stackbase+(x)])
#define ArgType(x) (stacktype[stackbase+(x)])
#define ArgCount (stackptr)
#define Shift(x) (ShiftLock(x)-((*mem<3||*mem>8)?2:0))
#define ShiftLock(x) (6-((row-(x)+3)%3))

static int run_line(char*text,int lp);
static void run_input(FILE*fp);

static inline void make_checksums(void) {
  int i,c,h;
  h=2;
  if(*mem>3) h=4;
  if(*mem>7) h=8;
  if(*mem>9) h=16;
  here+=here&(h-1);
  for(c=0,i=64;i<here;i++) c+=mem[i];
  mem[0x1A]=(here/h)>>8;
  mem[0x1B]=(here/h)&255;
  mem[0x1C]=(c>>8)&255;
  mem[0x1D]=c&255;
}

static Macro*find_mac(const char*name) {
  Macro*m=macro;
  int n=strcspn(name," \t\r\n;");
  while(m) {
    if(!strncasecmp(name,m->name,n) && strlen(m->name)==n) return m;
    m=m->next;
  }
  m=malloc(sizeof(Macro));
  m->name=strdup(name);
  m->name[n]=0;
  m->kind=0;
  m->next=macro;
  return macro=m;
}

static int in_row(const char*t,char r,char v) {
  const char*l;
  const char*h;
  int q=6;
  if(v==' ') return 0;
  if(v=='\r' && *mem==1) return 1;
  if(v=='\r' && r==2) return 7;
  if(r==0) l=t,h=t+26;
  if(r==1) l=t+26,h=t+52;
  if(r==2) l=t+54,h=t+78,q=8;
  for(t=l;t<h;t++,q++) if(*t==v) return q;
  return 0;
}

static inline void fill_in_fwords(const char*text,byte*beststate) {
  // Intelligent design sort is used (i.e. it is assumed the data is input in whatever order it needs to be in).
  const char*t;
  int i,j;
  for(i=0;i<synonyms;i++) {
    t=text-1;
    j=strlen(syntext[i])-1;
    while(t=strstr(t+1,syntext[i]))
      if(!beststate[t-text] && !beststate[t+j-text]) beststate[t-text]=i+4,memset(beststate+1+(t-text),255,j);
  }
}

static byte*cost_text(const char*text,const char*alpha,const byte*costtab,int use_syn) {
  unsigned int s1,s2,i,j,n;
  const char*t=text;
  static byte beststate[8192];
  byte*b=beststate;
  int ir[3]; // Alphabet in row?
  unsigned int costs[3];
  unsigned int next[3];
  static struct { byte data[8192]; } bests[6];
  memset(beststate,0,strlen(text));
  if(use_syn) fill_in_fwords(text,beststate);
  if(compliant && *mem>=3 && *mem<=8) return beststate;
  // Store character states
  while(*t) {
    if(*b) { // Contains fwords
      b++;
      t++;
      continue;
    }
    ir[0]=in_row(alpha,0,*t);
    ir[1]=in_row(alpha,1,*t);
    ir[2]=in_row(alpha,2,*t);
    if(*t==32 || (*mem==1 && *t==13) || (ir[0] && ir[1] && ir[2])) {
      *b++=3;
      t++;
      continue;
    }
    *b++=ir[0]?0:ir[1]?1:2;
    t++;
  }
  b=beststate;
  t=text;
  n=0;
  costs[0]=0;
  costs[1]=costs[2]=9;
  // Calculate best states
  while(*t) {
    if(*b<3) {
      for(s2=0;s2<3;s2++) {
        next[s2]=-1;
        for(s1=0;s1<3;s1++) {
          i=costtab[*b*9+s1*3+s2]+costs[s1];
          if(next[s2]>i) {
            next[s2]=i;
            j=s1;
          }
        }
        bests[s2+3]=bests[j];
        bests[s2+3].data[n]=j;
      }
      for(i=0;i<3;i++) {
        costs[i]=next[i];
        bests[i]=bests[i+3];
      }
    } else {
      bests[0].data[n]=bests[1].data[n]=bests[2].data[n]=*b;
    }
    n++;
    b++;
    t++;
  }
  // Select the best final list of states
  j=-1;
  for(i=0;i<3;i++) {
    if(j>costs[i]) {
      b=bests[s2=i].data;
      j=costs[i];
    }
  }
  // Move everything to left
  j=-1;
  for(i=0;text[i];i++) {
    if(b[i]<3) {
      if(j!=-1) b[j]=b[i];
      b[j=i]=s2;
    }
  }
  return bests[s2].data;
}

static void pack_text(const char*text,int use_syn,int is_dict,int is_direct) {
  static byte zch[8192];
  byte*beststate; // 0,1,2=state 3=keep 255=skip
  byte*zp=zch;
  byte*p;
  const char*t=text;
  int row=0;
  int i,j;
  int ir[3];
  const char*alpha=alphabet_table;
  const char*statev=(*mem<3 || *mem>8)?state_v1:state_v3;
  const char*u;
  if(*mem==1) {
    alpha=alphabet_table_old;
    use_syn=0;
  }
  if(*mem>=5 && (mem[0x34] || mem[0x35])) alpha=mem+(((mem[0x34]<<8)|mem[0x35])<<(*mem==10?2:*mem==9));
  if(is_dict) {
    while(*t) {
      if(i=in_row(alpha,0,*t)) {
        *zp++=i;
      } else if(i=in_row(alpha,1,*t)) {
        *zp++=Shift(1);
        *zp++=i;
      } else if(i=in_row(alpha,2,*t)) {
        *zp++=Shift(2);
        *zp++=i;
      } else {
        *zp++=Shift(2);
        *zp++=6;
        *zp++=((byte)*t)>>5;
        *zp++=(*t)&31;
      }
      t++;
    }
  } else {
    beststate=cost_text(text,alpha,(*mem<3 || *mem>8)?cost_v1:cost_v3,use_syn);
    j=0;
    for(i=0;text[i];i++) {
      if(beststate[i]==3) beststate[i]=j;
      if(beststate[i]<3) {
        ir[0]=in_row(alpha,0,text[i]);
        ir[1]=in_row(alpha,1,text[i]);
        ir[2]=in_row(alpha,2,text[i]);
        row=(text[i]==32 || (text[i]==13 && *mem==1))?j:ir[2]?2:ir[1]?1:ir[0]?0:2;
        u=statev+row*54+j*18+beststate[i]*6;
        while(*u!='-') {
          if(*u=='*') {
            if(text[i]==32) {
              *zp++=0;
            } else if(text[i]==13 && *mem==1) {
              *zp++=1;
            } else if((statev==state_v3 && j==2 && beststate[i]==2 && (row!=2 || !ir[2])) || !(ir[0] || ir[1] || ir[2])) {
              *zp++=6;
              *zp++=((byte)text[i])>>5;
              *zp++=text[i]&31;
            } else {
              *zp++=ir[row];
            }
          } else {
            *zp++=*u&7;
          }
          u++;
        }
        j=beststate[i];
      } else if(beststate[i]!=255) {
        *zp++=((beststate[i]-4)>>5)+1;
        *zp++=(beststate[i]-4)&31;
      }
    }
  }
  for(i=0;i<(is_dict?9:3);i++) zp[i]=5;
  if(is_dict) zp=zch+(*mem<4?6:9);
  if(is_direct) {
    for(p=zch;p<zp;p+=3) mem[here++]=(p[0]<<2)|(p[1]>>3),mem[here++]=(p[1]<<5)|p[2];
    mem[here-2]|=0x80;
  } else {
    for(p=zch;p<zp;p+=3) Push((p[0]<<10)|(p[1]<<5)|p[2]);
    stack[stackbase+stackptr-1]|=0x8000;
  }
}

static void make_object(int id) {
  int x;
  while(obj_count<=id) {
    obj_data=realloc(obj_data,sizeof(Object)*(obj_count+1));
    memset(obj_data+obj_count,0,sizeof(Object));
    obj_count++;
  }
}

static void write_branch_offset(int br,int inv) {
  int o=br-here;
  if(br==0 || br==1) {
    mem[here++]=(inv<<7)|br|0x40;
  } else if(o>0 && o<=62) {
    mem[here++]=(inv<<7)|(o+1)|0x40;
  } else {
    o&=0x3FFF;
    mem[here++]=(inv<<7)|(o>>8);
    mem[here++]=o&255;
  }
}

static void compile_instruction(Macro*inst) {
  byte op[8]={3,3,3,3,3,3,3,3};
  int*s=stack+stackbase+(inst->kind==11);
  byte*t=stacktype+stackbase+(inst->kind==11);
  byte ic=inst->kind==11?*s:inst->data&255;
  int ex=inst->kind==11?inst->cmd==78:inst->data&256;
  int tx=inst->kind==11?*s<0:inst->data&512;
  int cnt=stackptr-(inst->kind==11);
  int i;
  int br_use=0;
  int st_use=0;
  int inv=1;
  int br,st;
  if(inst->kind==11 && *s<0) ic=-*s;
  for(i=0;i<cnt && i<8;i++) {
    if(t[i]&1) inv=0;
    if(t[i]&2) {
      st_use=1;
      st=s[i];
    }
    if(t[i]&4) op[i]=2;
    else if(s[i]&~255) op[i]=0;
    else op[i]=1;
  }
  cnt-=st_use;
  //if(((ex || ic>=0xE0) && tx) || (((ic<0x20 && cnt==3 && ic!=27) || (ic>=0x20 && ic<0xB0 && cnt==2) || (ic>=0xB0 && ic<0xC0 && cnt==1)) && !(ex || tx))) {
  if(tx) {
    br_use=1;
    br=s[--cnt];
  }
  for(i=cnt;i<8;i++) op[i]=3;
  if(ex) {
    mem[here++]=0xBE; // Prefix for extended instructions
  } else if(ic<0x20 && (op[0]==0 || op[1]==0 || op[2]!=3)) {
    ic|=0xC0; // Encode a 2OP instruction in VAR form
  }
  if(ex || ic>=0xC0) {
    // VAR
    mem[here++]=ic;
    mem[here++]=(op[0]<<6)|(op[1]<<4)|(op[2]<<2)|op[3];
    if(ic==236 || ic==250) mem[here++]=(op[4]<<6)|(op[5]<<4)|(op[6]<<2)|op[7];
  } else if(ic<0x20) {
    // 2OP
    mem[here++]=ic|((op[0]&2)<<5)|((op[1]&2)<<4);
    cnt=2;
  } else if(ic<0xB0) {
    // 1OP
    mem[here++]=ic|(*op<<4);
    cnt=1;
  } else {
    // 0OP
    mem[here++]=ic;
    if(cnt && !(br_use || st_use)) { //if(tx) {
      while(s<stack+stackbase+stackptr) {
        mem[here++]=*s>>8;
        mem[here++]=*s++&255;
      }
    }
    cnt=0;
  }
  for(i=0;i<cnt;i++) {
    if(!op[i]) mem[here++]=(s[i]>>8)&255;
    mem[here++]=s[i]&255;
  }
  if(st_use) mem[here++]=st;
  if(br_use) write_branch_offset(br,inv);
}

static void run_macro(char*text) {
  char buf[4096];
  char*p=buf;
  int os=stackbase;
  int om=mac_args;
  int lp=1;
  mac_args=stackptr;
  stackbase+=stackptr;
  stackptr=0;
  while(*text) {
    if(*text=='\n') {
      *p++='\n';
      *p++=0;
      text++;
      if(run_line(buf,lp)) break;
      p=buf;
      if(*buf!='\n' && *buf!=';') lp=0;
    } else {
      *p++=*text++;
    }
  }
  stackptr+=stackbase-os;
  stackbase=os;
  mac_args=om;
}

static void run_primitive(char cmd) {
  int x,y,z;
  FILE*fp;
  switch(cmd) {
    case 1: // +
      y=Pop; x=Pop;
      Push(x+y);
      break;
    case 2: // -
      y=Pop; x=Pop;
      Push(x-y);
      break;
    case 3: // *
      y=Pop; x=Pop;
      Push(x*y);
      break;
    case 4: // /
      y=Pop; x=Pop;
      Push(x/y);
      break;
    case 5: // 1+
      x=Pop;
      Push(x+1);
      break;
    case 6: // 1-
      x=Pop;
      Push(x-1);
      break;
    case 7: // AND_
      y=Pop; x=Pop;
      Push(x&y);
      break;
    case 8: // OR_
      y=Pop; x=Pop;
      Push(x|y);
      break;
    case 9: // DROP
      Pop;
      break;
    case 10: // DUP
      x=Pop;
      Push(x); Push(x);
      break;
    case 11: // DEPTH
      x=stackptr;
      Push(x);
      break;
    case 12: // @
      x=Pop;
      Push((mem[x]<<8)|mem[x+1]);
      break;
    case 13: // !
      x=Pop; y=Pop;
      mem[x]=y>>8;
      mem[x+1]=y&255;
      break;
    case 14: // C@
      x=Pop;
      Push(mem[x]);
      break;
    case 15: // C!
      x=Pop; y=Pop;
      mem[x]=y;
      break;
    case 16: // HERE
      Push(here);
      break;
    case 17: // PASS
      Push(pass);
      break;
    case 18: // NOT_
      x=Pop;
      Push(~x);
      break;
    case 19: // <
      y=Pop; x=Pop;
      Push(-(x<y));
      break;
    case 20: // >
      y=Pop; x=Pop;
      Push(-(x>y));
      break;
    case 21: // =
      y=Pop; x=Pop;
      Push(-(x==y));
      break;
    case 22: // SWAP
      y=Pop; x=Pop;
      Push(y); Push(x);
      break;
    case 23: // ,
      mem[here++]=Pop;
      break;
    case 24: // ARGS
      for(x=0;x<mac_args;x++) Type(stacktype[stackbase+x-mac_args]),Push(stack[stackbase+x-mac_args]);
      break;
    case 25: // ARG@
      x=Pop;
      Type(stacktype[stackbase+x-mac_args]);
      Push(stack[stackbase+x-mac_args]);
      break;
    case 26: // ARG!
      x=Pop; y=Pop;
      stack[stackbase+x-mac_args]=y;
      break;
    case 27: // ARGC
      Push(mac_args);
      break;
    case 28: // CODE
      Type((label->data>>8)&1);
      Push(label->data&255);
      break;
    case 29: // I (invert)
      Type(1);
      break;
    case 30: // S (store)
      Type(2);
      break;
    case 31: // V (variable)
      Type(4);
      break;
    case 32: // I?
      x=stacktype[stackbase+stackptr-1];
      Push(-(x&1));
      break;
    case 33: // S?
      x=stacktype[stackbase+stackptr-1];
      Push(-(x&2));
      break;
    case 34: // V?
      x=stacktype[stackbase+stackptr-1];
      Push(-(x&4));
      break;
    case 35: // DEF?
      Push(-!!label->kind);
      break;
    case 36: // LSH
      y=Pop; x=Pop;
      Push(x<<y);
      break;
    case 37: // RSH
      y=Pop; x=Pop;
      Push(x>>y);
      break;
    case 39: // EXP@
      x=Pop;
      Type(stacktype[stackbase-x]);
      Push(stack[stackbase-x]);
      break;
    case 40: // EXP!
      x=Pop; y=Pop;
      stack[stackbase-x]=y;
      break;
    case 41: // RANGE@
      x=Pop; y=Pop;
      while(x<y) Push(mem[x++]);
      break;
    case 42: // LOC@
      x=Pop;
      make_object(x);
      Push(obj_data[x].loc);
      break;
    case 43: // NAME@
      x=Pop;
      make_object(x);
      if(obj_data[x].name) for(y=0;obj_data[x].name[y];y++) Push(obj_data[x].name[y]);
      break;
    case 65: // MACRO
      if(label->kind==1) free(label->text);
      label->kind=1;
      label->text=strdup("");
      defining=1;
      break;
    case 67: // IF
      skipping=!Arg(0);
      break;
    case 70: // IFDEF
      skipping=!label->kind;
      break;
    case 72: // IS
      if(label1) label1->data=Arg(0);
      break;
    case 73: // ORG
      here=Arg(0)&0xFFFFF;
      break;
    case 79: // MAKE_0OP
    case 80: // MAKE_1OP
    case 81: // MAKE_2OP
    case 82: // MAKE_VAR
    case 83: // MAKE_EXT
      label->kind=8;
      if(Arg(0)<0) label->data=0x200|-Arg(0);
      else label->data=Arg(0);
      if(cmd==83) label->data|=0x100;
      break;
    case 85: // DB
      for(x=0;x<ArgCount;x++) mem[here++]=Arg(x);
      break;
    case 86: // DW
      for(x=0;x<ArgCount;x++) mem[here++]=Arg(x)>>8,mem[here++]=Arg(x)&255;
      break;
    case 87: // INCLUDE
      fp=fopen(label->name,"r");
      if(fp) {
        run_input(fp);
        fclose(fp);
      } else {
        fprintf(stderr,"File not found: %s\n",label->name);
      }
      break;
    case 88: // SYNONYMS
      here+=here&1;
      y=(mem[0x18]<<8)|mem[0x19];
      if(*mem==9) y<<=1;
      if(*mem==10) y<<=2;
      for(x=dynfwords;x<synonyms;x++) {
        mem[y+(x<<1)]=here>>9;
        mem[y+(x<<1)+1]=here>>1;
        pack_text(syntext[x],0,0,1);
      }
      break;
    case 89: // ERROR
      for(x=0;x<ArgCount;x++) {
        if(ArgType(x)) fprintf(stderr," %d (%08X) ",Arg(x),Arg(x));
        else fputc(Arg(x),stderr);
      }
      fputc('\n',stderr);
      fflush(stderr);
      break;
    case 90: // ALIGNPACKED
      x=2;
      if(*mem>3) x=4;
      if(*mem>7) x=8;
      if(*mem>9) x=16;
      here+=(-here)&(x-1);
      if(label1) label1->data=here;
      break;
    case 91: // DW_SKIP
      for(x=Arg(0)+1;x<ArgCount;x++) mem[here++]=Arg(x)>>8,mem[here++]=Arg(x)&255;
      break;
    case 92: // DS
      for(x=0;x<Arg(0);x++) mem[here++]=0;
      break;
    case 95: // MCOPY
      if(label_clip.kind==1) free(label_clip.text);
      label_clip=*label;
      if(label_clip.kind==1) label_clip.text=strdup(label_clip.text);
      break;
    case 96: // MPASTE
      label->kind=label_clip.kind;
      if(label_clip.kind==1) label->text=label_clip.text;
      else if(label_clip.kind==2 || label_clip.kind==8) label->data=label_clip.data;
      else label->cmd=label_clip.cmd;
      break;
    case 97: // LONG_TEXT
      long_text_macro=label;
      long_text_ptr=long_text_data;
      break;
    case 98: // DYNFWORDS
      synonyms=dynfwords=Arg(0);
      for(x=0;x<dynfwords;x++) free(syntext[x]),sprintf(syntext[x]=malloc(5),"%c%d%c",x);
      break;
    case 99: // INC_BIN
      fp=fopen(label->name,"rb");
      if(fp) {
        fseek(fp,0,SEEK_END);
        x=ftell(fp);
        rewind(fp);
        fread(mem+here,x,1,fp);
        here+=(unsigned int)x;
        fclose(fp);
      } else {
        fprintf(stderr,"File not found: %s\n",label->name);
      }
      break;
    case 100: // COMPLIANT
      compliant=ArgCount?Arg(0):1;
      break;
    case 101: // Z-ENCODE
      for(x=0;x<ArgCount;x++) long_text_data[x]=Arg(x);
      long_text_data[ArgCount]=0;
      pack_text(long_text_data,1,0,1);
      break;
    case 102: // MAXPASS
      maxpass=ArgCount?(Arg(0)>69105?69105:Arg(0)):6;
      break;
    case 103: // MACLOOP
      macloop=Arg(0);
      break;
    case 104: // MRUNLINE
      for(x=0;x<ArgCount;x++) long_text_data[x]=Arg(x);
      long_text_data[ArgCount]=0;
      {
        char*i=strdup(long_text_data);
        run_line(i,1);
        free(i);
      }
      break;
    case 105: // EMIT_OBJECTS
      x=(min_property_defaults-1)<<1;
      x=(here-x)>>(*mem==10?2:*mem==9);
      if(*mem==10) here+=(-here)&3;
      if(*mem==9) here+=(-here)&1;
      mem[0x0A]=x>>8;
      mem[0x0B]=x&255;
      for(x=min_property_defaults;x<(*mem<4?32:64);x++) {
        mem[here++]=property_defaults[x]>>8;
        mem[here++]=property_defaults[x]&255;
      }
      if(*mem<4) {
        for(x=1;x<obj_count;x++) {
          memcpy(mem+here,obj_data[x].attr,4);
          here+=4;
          mem[here++]=obj_data[x].loc;
          mem[here++]=obj_data[x].next;
          mem[here++]=obj_data[x].first;
          mem[here++]=obj_data[x].table>>8;
          mem[here++]=obj_data[x].table&255;
        }
      } else {
        for(x=1;x<obj_count;x++) {
          memcpy(mem+here,obj_data[x].attr,6);
          here+=6;
          mem[here++]=obj_data[x].loc>>8;
          mem[here++]=obj_data[x].loc&255;
          mem[here++]=obj_data[x].next>>8;
          mem[here++]=obj_data[x].next&255;
          mem[here++]=obj_data[x].first>>8;
          mem[here++]=obj_data[x].first&255;
          mem[here++]=obj_data[x].table>>8;
          mem[here++]=obj_data[x].table&255;
        }
      }
      break;
    case 106: // EMIT_PROPERTIES
      z=(mem[0x0A]<<8)|mem[0x0B];
      if(*mem==9) z<<=1;
      if(*mem==10) z<<=2;
      if(*mem<4) {
        z+=53;
        //TODO
      } else {
        z+=112;
        for(x=1;x<obj_count;x++) {
          if(!obj_data[x].table) {
            if(*mem==10) here+=(-here)&3;
            if(*mem==9) here+=(-here)&1;
            y=here>>(*mem==10?2:*mem==9);
            mem[z+x*14+12]=y>>8;
            mem[z+x*14+13]=y&255;
            if(obj_data[x].name) {
              y=here++;
              pack_text(obj_data[x].name,1,0,1);
              mem[y]=(here-y)>>1;
            } else {
              mem[here++]=0;
            }
            for(y=127;y;y--) {
              if(obj_data[x].size[y]) {
                if(obj_data[x].size[y]>2 || (y&64)) {
                  mem[here++]=y|0x80;
                  mem[here++]=obj_data[x].size[y]|0xC0;
                } else {
                  mem[here++]=y|(obj_data[x].size[y]==2?64:0);
                }
                memcpy(mem+here,obj_data[x].data[y],obj_data[x].size[y]);
                here+=obj_data[x].size[y];
              }
            }
            mem[here++]=0;
          }
        }
      }
      break;
    case 107: // ATTRIBUTE+
    case 108: // ATTRIBUTE-
      make_object(Arg(0));
      for(x=1;x<ArgCount;x++) {
        if((!ArgType(x))^(cmd&1)) obj_data[Arg(0)].attr[Arg(x)>>3]&=~(128>>(Arg(x)&7));
        else obj_data[Arg(0)].attr[Arg(x)>>3]|=128>>(Arg(x)&7);
      }
      break;
    case 109: // SDESC
      make_object(Arg(0));
      if(obj_data[Arg(0)].name) free(obj_data[Arg(0)].name);
      obj_data[Arg(0)].name=malloc(ArgCount);
      for(x=1;x<ArgCount;x++) obj_data[Arg(0)].name[x-1]=Arg(x);
      obj_data[Arg(0)].name[ArgCount-1]=0;
      break;
    case 110: // LOCATION
      make_object(x=Arg(0));
      if(pass && obj_data[x].loc) break;
      make_object(y=Arg(1));
      obj_data[x].loc=y;
      if(pass) obj_data[y].next=obj_data[y].first;
      if(pass) obj_data[y].first=x;
      break;
    case 111: // XPROPERTY
      make_object(x=Arg(0));
      obj_data[x].table=here>>(*mem==10?2:*mem==9);
      break;
    case 112: // BPROPERTY
      make_object(x=Arg(0));
      y=Arg(1);
      if(obj_data[x].data[y]) free(obj_data[x].data[y]);
      obj_data[x].data[y]=malloc(obj_data[x].size[y]=ArgCount-2);
      for(z=0;z<ArgCount-2;z++) obj_data[x].data[y][z]=Arg(z+2);
      break;
    case 113: // WPROPERTY
      make_object(x=Arg(0));
      y=Arg(1);
      if(obj_data[x].data[y]) free(obj_data[x].data[y]);
      obj_data[x].data[y]=malloc(obj_data[x].size[y]=(ArgCount-2)<<1);
      for(z=0;z<ArgCount-2;z++) obj_data[x].data[y][z<<1]=Arg(z+2)>>8,obj_data[x].data[y][(z<<1)|1]=Arg(z+2)&255;
      break;
    case 114: // DEFAULT
      if(min_property_defaults>Arg(0)) min_property_defaults=Arg(0);
      property_defaults[Arg(0)&63]=Arg(1);
      break;
    case 115: // ENCODING
      if(Arg(0)=='C') {
        encoding=1; // CP437/CP850
      } else if(Arg(0)=='U') {
        encoding=2; // UTF-8
      } else if(Arg(0)=='W' || Arg(0)=='I') {
        encoding=3; // Windows-1252/ISO-8859-1
      } else {
        encoding=0; // ZSCII/ASCII
      }
      break;
  }
}

static void run_line_2(char*text) {
  Macro*m;
  int x;
  char buf[4095];
  char c;
  char*p;
  for(;;) {
    x=0;
    while(*text && *text<=' ') text++;
    if(*text==';' || !*text) return;
    if(*text=='"' || *text=='\'' || *text=='[' || *text=='(' || *text=='{') {
      c=*text++;
      if(c=='[') c=']';
      if(c=='(') c=')';
      if(c=='{') c='}';
      p=buf;
      while(*text!=c) {
        if(*text=='\\') *p++=*++text;
        else if(*text=='^') *p++='\r';
        else if(*text=='~') *p++=c;
        else *p++=*text;
        text++;
      }
      *p=0;
      if(c=='\'') for(p=buf;*p;p++) Push(*p&255);
      if(c=='"') pack_text(buf,1,0,0);
      if(c==']') pack_text(buf,0,1,0);
      if(c==')') pack_text(buf,0,0,0);
      if(c=='}') free(syntext[synonyms]),syntext[synonyms++]=strdup(buf);
      text++;
    } else if(*text>='0' && *text<='9' && (text[1]<=' ' || (text[1]>='0' && text[1]<='9'))) {
      while(*text>='0' && *text<='9') x=10*x+(*text++)-'0';
      Push(x);
    } else if(*text=='-' && text[1]>='0' && text[1]<='9') {
      text++;
      while(*text>='0' && *text<='9') x=10*x+(*text++)-'0';
      Push(-x);
    } else if(*text=='$' && text[1]=='$') {
      text+=2;
      while(*text=='0' || *text=='1') x=(x<<1)|(*text++&1);
      Push(x);
    } else if(*text=='$') {
      text++;
      while(*text>' ' && *text!=';') {
        x<<=4;
        if(*text>='0' && *text<='9') x|=*text&15;
        else if((*text>='A' && *text<='F') || (*text>='a' && *text<='f')) x|=(*text+10-'a')&15;
        text++;
      }
      Push(x);
    } else {
      m=find_mac(text);
      while(*text>' ') text++;
      switch(m->kind) {
        case 0:
          Push(0);
          break;
        case 1:
          do run_macro(m->text); while(macloop);
          break;
        case 2:
          Push(m->data);
          break;
        case 6:
          run_primitive(m->cmd);
          break;
        case 9:
          while(*text && *text<=' ') text++;
          label=find_mac(text);
          while(*text>' ') text++;
          run_primitive(m->cmd);
          break;
        case 12:
          Push(block_addr[block_count]);
          block_stack[block_ptr++]=block_count++;
          break;
        case 13:
          Push(block_addr[block_stack[--block_ptr]]);
          break;
        default:
          fprintf(stderr,"Wrong kind of symbol here: %s\n",m->name);
          break;
      }
    }
  }
}

static int run_line(char*text,int lp) {
  Macro*m;
  int i;
  if(!lp) label1=0;
run_line_again:
  if(*text=='\n' || *text==';' || !*text) return 0;
  stackptr=0;
  m=find_mac(text);
  if(*text==':') m->kind=2;
  if(!defining) {
    while(*text>' ') text++;
    text+=!!*text;
  }
  if(skipping) {
    if(m->kind==3) skipping++;
    if(m->kind==4 && skipping==1) skipping=0;
    if(m->kind==5) skipping--;
  } else if(defining) {
    if(m->kind==7) {
      defining=0;
    } else {
      label->text=realloc(label->text,strlen(label->text)+strlen(text)+2);
      strcat(label->text,text);
    }
  } else switch(m->kind) {
    case 0:
      fprintf(stderr,"Undefined symbol: %s\n",m->name);
      break;
    case 1:
      do {
        run_line_2(text);
        run_macro(m->text);
      } while(macloop);
      break;
    case 2:
      m->data=here;
      label=label1=m;
      goto run_line_again;
    case 3:
      if(m->cmd==67) run_line_2(text); else label=find_mac(text);
      run_primitive(m->cmd);
      break;
    case 4:
      skipping=1;
      break;
    case 5:
      // Do Nothing
      break;
    case 6:
      run_line_2(text);
      run_primitive(m->cmd);
      break;
    case 7:
      fprintf(stderr,"ENDMAC found prematurely\n");
      break;
    case 8:
      run_line_2(text);
      compile_instruction(m);
      break;
    case 9:
      label=find_mac(text);
      run_primitive(m->cmd);
      break;
    case 10:
      run_line_2(text);
      return 1;
    case 11:
      run_line_2(text);
      compile_instruction(m);
      break;
    case 12:
      block_addr[block_count]=here;
      block_stack[block_ptr++]=block_count++;
      goto run_line_again;
    case 13:
      block_addr[block_stack[--block_ptr]]=here;
      goto run_line_again;
  }
  stackptr=0;
  for(i=0;i<16;i++) stacktype[stackbase+i]=0;
  return 0;
}

static inline void run_long_text(char*p) {
  // Long text macro types: !normal &paragraph
  char*q=p+strlen(p)-1;
  if(*p && *q=='\n') *q--=0;
  if(*p && *q=='\r') *q--=0;
  while(q>p && *q==' ') *q--=0;
  if(strcmp(long_text_macro->name,p)) {
    if(long_text_ptr!=long_text_data) *long_text_ptr++='\r';
    if(*long_text_macro->name=='&') {
      while(*p==' ' || *p=='\t') p++;
      if(long_text_ptr>long_text_data+1 && long_text_ptr[-1]=='\r' && long_text_ptr[-2]!='\r') long_text_ptr[-1]=' ';
    }
    if(!*p) return;
    strcpy(long_text_ptr,p);
    while(*++long_text_ptr);
  } else {
    if(*long_text_macro->name=='&') *long_text_ptr++='\r';
    *long_text_ptr=0;
    long_text_macro=0;
    pack_text(long_text_data,1,0,1);
  }
}

static inline void decode_input(char*p) {
  int utf[97]={0xE4,0xF6,0xFC,0xC4,0xD6,0xDC,0xDF,0xBB,0xAB,0xEB,0xEF,0xFF,0xCB,0xCF,0xE1,0xE9,0xED,0xF3,
   0xFA,0xFD,0xC1,0xC9,0xCD,0xD3,0xDA,0xDD,0xE0,0xE8,0xEC,0xF2,0xF9,0xC0,0xC8,0xCC,0xD2,0xD9,0xE2,0xEA,0xEE,0xF4,0xFB,
   0xC2,0xCA,0xCE,0xD4,0xDB,0xE5,0xC5,0xF8,0xD8,0xE3,0xF1,0xF5,0xC3,0xD1,0xD5,0xE6,0xC6,0xE7,0xC7,0xFE,0xF0,0xDE,0xD0,
   0xA3,0x153,0x152,0xA1,0xBF,[69 ... 96]=0xFFFF};
  static const byte ibm[128]={
   214,157,170,191,155,181,201,213,192,164,182,165,193,183,158,202,
   176,211,212,194,156,184,195,185,166,159,160,203,219,204,'x','?',
   169,171,172,173,206,209,'a','o',223,'?','?','?','?',222,163,162,
   'Z','_','?',')','-',175,196,186,'C','?','?','?','?','c','Y','0',
   '.','*','+',',','&','[',205,208,'?','?','?','?','?','?','?','?',
   216,218,197,167,187,'i',177,198,168,'1','/','6','8','9',188,'7',
   178,161,199,189,207,210,'u',215,217,179,200,190,174,180,'K','?',
   '-','+','L','?','?','?','?','?','?','?','?','?','n','?','W',' '};
  static const byte win[128]={
   '?','?','?','?','?','?','?','?','^','?','S','<',221,'?','Z','?',
   '?','?','?','"','"','?','-','-','~','?','s','>',220,'?','z','Y',
   ' ',222,'c',219,'?','Y','|','S','?','C','a',163,'?','-','R','K',
   '?','?','?','?','?','u','?','?','?','?','o',162,'?','?','?',223,
   186,175,196,208,158,202,212,214,187,176,197,167,188,177,198,168,
   218,209,189,178,199,210,159,'?',204,190,179,200,160,180,217,161,
   181,169,191,205,155,201,211,213,182,170,192,164,183,171,193,165,
   216,206,184,172,194,207,156,'?',203,185,173,195,157,174,215,166};
  int i,j,m,n;
  char*q=p;
  if(encoding==2 && compliant && *mem>=5 && *mem<=8 && (mem[0x36] || mem[0x37])) {
    i=(mem[0x36]<<8)|mem[0x37];
    if(mem[i+1]>=3 && (mem[i+6] || mem[i+7])) {
      i=((mem[i+6]<<8)|mem[i+7])+1;
      for(j=0;j<97;j++) utf[j]=(mem[i+(j<<1)]<<8)|mem[i+1+(j<<1)];
    }
  }
  while(*p) {
    if(*p&128) {
      if(encoding==2) {
        if(*p&64) {
          q=p;
          m=*p&32,n=*p&31; // Only up to 16-bits characters can be used
        } else {
          n=(n<<6)|(*p&63);
          if(!m) {
            memmove(q,p,strlen(p)+1);
            p=q;
            for(i=0;i<97;i++) if(n==utf[i]) *p=i+155;
          }
          m=0;
        }
      } else {
        *p=(encoding==1?ibm:win)[*p&127];
      }
    }
    p++;
  }
}

static void run_input(FILE*fp) {
  char buf[4096];
  char*p;
  byte e=encoding;
  encoding=0;
  while(!feof(fp)) {
    *buf=0;
    fgets(p=buf,4095,fp);
    if(encoding) decode_input(buf);
    if(long_text_macro) {
      run_long_text(p);
    } else {
      while(*p && *p<=' ') p++;
      run_line(p,0);
    }
  }
  encoding=e;
}

static void primitive(const char*name,char kind,char cmd) {
  Macro*m=find_mac(name);
  m->kind=kind;
  m->cmd=cmd;
}

int main(int argc,char**argv) {
  int i;
#ifdef _WIN32
  _setmode(_fileno(stdin),_O_BINARY);
  _setmode(_fileno(stdout),_O_BINARY);
#endif
  primitive("+",6,1);
  primitive("-",6,2);
  primitive("*",6,3);
  primitive("/",6,4);
  primitive("1+",6,5);
  primitive("1-",6,6);
  primitive("AND_",6,7);
  primitive("OR_",6,8);
  primitive("DROP",6,9);
  primitive("DUP",6,10);
  primitive("DEPTH",6,11);
  primitive("@",6,12);
  primitive("!",6,13);
  primitive("C@",6,14);
  primitive("C!",6,15);
  primitive("HERE",6,16);
  primitive("PASS",6,17);
  primitive("NOT_",6,18);
  primitive("<",6,19);
  primitive(">",6,20);
  primitive("=",6,21);
  primitive("SWAP",6,22);
  primitive(",",6,23);
  primitive("ARGS",6,24);
  primitive("ARG@",6,25);
  primitive("ARG!",6,26);
  primitive("ARGC",6,27);
  primitive("CODE",9,28);
  primitive("I",6,29);
  primitive("S",6,30);
  primitive("V",6,31);
  primitive("I?",6,32);
  primitive("S?",6,33);
  primitive("V?",6,34);
  primitive("DEF?",9,35);
  primitive("LSH",6,36);
  primitive("RSH",6,37);
  primitive("AS",9,38);
  primitive("EXP@",6,39);
  primitive("EXP!",6,40);
  primitive("RANGE@",6,41);
  primitive("LOC@",6,42);
  primitive("NAME@",6,43);
  primitive("MACRO",9,65);
  primitive("ENDMAC",7,66);
  primitive("IF",3,67);
  primitive("ELSE",4,68);
  primitive("ENDIF",5,69);
  primitive("IFDEF",3,70);
  primitive("CALC",6,71);
  primitive("IS",6,72);
  primitive("ORG",6,73);
  primitive("0OP",11,74);
  primitive("1OP",11,75);
  primitive("2OP",11,76);
  primitive("VAR",11,77);
  primitive("EXT",11,78);
  primitive("MAKE_0OP",6,79);
  primitive("MAKE_1OP",6,80);
  primitive("MAKE_2OP",6,81);
  primitive("MAKE_VAR",6,82);
  primitive("MAKE_EXT",6,83);
  primitive("MRETURN",10,84);
  primitive("DB",6,85);
  primitive("DW",6,86);
  primitive("INCLUDE",9,87);
  primitive("SYNONYMS",6,88);
  primitive("ERROR",6,89);
  primitive("ALIGNPACKED",6,90);
  primitive("DW_SKIP",6,91);
  primitive("DS",6,92);
  primitive("<|",12,0);
  primitive("|>",13,0);
  primitive("MCOPY",9,95);
  primitive("MPASTE",9,96);
  primitive("LONG_TEXT",9,97);
  primitive("DYNFWORDS",6,98);
  primitive("INC_BIN",9,99);
  primitive("COMPLIANT",6,100);
  primitive("Z-ENCODE",6,101);
  primitive("MAXPASS",6,102);
  primitive("MACLOOP",6,103);
  primitive("MRUNLINE",6,104);
  primitive("EMIT_OBJECTS",6,105);
  primitive("EMIT_PROPERTIES",6,106);
  primitive("ATTRIBUTE+",6,107);
  primitive("ATTRIBUTE-",6,108);
  primitive("SDESC",6,109);
  primitive("LOCATION",6,110);
  primitive("XPROPERTY",6,111);
  primitive("BPROPERTY",6,112);
  primitive("WPROPERTY",6,113);
  primitive("DEFAULT",6,114);
  primitive("ENCODING",6,115);
  filin=stdin;
  filout=stdout;
  for(i=1;i<argc;i++) {
    if(argv[i][0]=='-') {
      switch(argv[i][1]) {
        case 'd':
          primitive(argv[++i],2,1);
          break;
        case 'i':
          filin=fopen(argv[++i],"r");
          break;
        case 'o':
          filout=fopen(story_name=argv[++i],"wb");
          break;
        case 'v':
          puts("Frolg " VERSION);
          return 1;
        default:
          fprintf(stderr,"Unknown switch: %c\n",argv[i][1]);
      }
    } else if(argv[i][0]) {
      fprintf(stderr,"Wrong command-line arguments: %s\n",argv[i]);
      return 1;
    }
  }
  for(pass=0;pass<maxpass;pass++) {
    fprintf(stderr,"Pass %d: %06X\n",pass,here);
    synonyms=0;
    here=0;
    block_count=0;
    block_ptr=0;
    run_input(filin);
    rewind(filin);
  }
  make_checksums();
  fwrite(mem,here,1,filout);
  return 0;
}
