#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>


/*
 *   SayTopic & Query Topic by Eric Eve Version 0.4 (18-Feb-2007)
 *
 *   This module adds two more types of TopicEntry: SayTopic and QueryTopic, 
 *   along with the associated classes DefaultSayTopic, SuggestedSayTopic, 
 *   DefaultQueryTopic and SuggestedQueryTopic. There are also two new hybrid
 *   classes AskQueryTopic and DefaultAskQueryTopic.
 *
 *   SayTopic can be used for making statements to the NPCs with commands 
 *   like SAY I LOVE TO DORIS or TELL DORIS THAT YOU LOVE HER. SayTopics can 
 *   be matched on Topic objects, but it may work better to match them on 
 *   regular expressions.
 *
 *   QueryTopic can be used for asking questions involving who, what, why, 
 *   when, if, how, which or whether, e.g.:
 *
 *   ASK BOB HOW HE IS
 *
 *   ASK FRED WHERE HE CAME FROM
 *
 *   Note that QueryTopic uses an additional property qType to determine the 
 *   stype of question asked (who, what, where etc.). To set up the previous 
 *   two examples using QueryTopic and SuggestedQueryTopic you might define:
 *
 *.   + QueryTopic, SuggestedQueryTopic 'he is$' 
 *.       "<q>How are you?</q> you  ask.\b 
 *.       <q>Pretty well,</q> he replies. " 
 *.       qType = 'how' 
 *.       name = 'how he is'
 *.   ;
 *
 *.   + QueryTopic, SuggestedQueryTopic 'he (came|comes) from$' 
 *.    "<q>Where do you come from, Bob?</q> you enquire.\b 
 *.     <q>London, originally,</q> he tells you. " 
 *.     qType = 'where' 
 *.     name = 'where he comes from'
 *    ;
 *
 *   Note that to reconstruct the complete question you need both 
 *   gAction.qType and gTopicText, e.g.:
 *
 *.   + DefaultQueryTopic 
 *.      "You ask Bob <<gAction.qType.toLower()>> <<gTopicText>>, but he 
 *.      declines to answer. " 
 *.   ;
 *
 *   Finally, both AskTopic and QueryTopic can be matched on a list of 
 *   keywords (like a SpecialTopic) in addition to the other types of match 
 *   (object and regular expression). Within such a list certain keywords 
 *   can be designated 'weak' by being enclosed in brackets (e.g. common 
 *   words like 'is' or 'an'); this prevents the topic being matched on such 
 *   words alone. You can also define whether to allow duplicate keywords to 
 *   match (e.g. ASK BOB IF FRED FRED IS WELL WELL WELL) and also whether to 
 *   enforce the order of keywords (e.g. to allow ASK IF FRED IS WELL or ASK 
 *   IF FRED WELL but disallow ASK IF WELL IS FRED). You can also define 
 *   alternative tokens by using a vertical bar (|) to separate them, e.g.:
 *
 *.   +QueryTopic, SuggestedQueryTopic ['where'] 
 *.      ['he|bob', '(comes|came)' 'from'] 
 *.      "<q>Where do you come from, Bob?</q> you enquire.\b 
 *.      <q>London, originally,</q> he tells you. " 
 *.       name = 'where he comes from'
 *.   ;
 *
 *   If you need anything tighter than this, it's probably best to use a 
 *   regular expression match instead.
 */

ModuleID
    name = 'SayTopic & QueryTopic'
    byLine = 'by Eric Eve'
    htmlByLine = 'by <A href="mailto:eric.eve@hmc.ox.ac.uk">Eric Eve</a>'
    version = '1.0'  
; 


class SayTopic: TopicMatchTopic
    includeInList = [&sayTopics]
    matchTopic(fromActor, topic)
    {
        if (matchObj != nil && dataType(topic) != TypeSString)
        {
            
            if (matchObj.ofKind(Collection))
            {
                /* try matching each object in the list */
                if (matchObj.indexWhich({x: findMatchObj(x, topic)}) != nil)
                    return matchScore;
            }
            else
            {
                /* match the single object */
                if (findMatchObj(matchObj, topic))
                    return matchScore;
            }
        }
        
        if(matchPattern == nil && keywordList == nil)
            return nil;
        
        
        local txt;
        
        /* 
         *   There's no match object; try matching our regular expression to 
         *   the actual topic text.  Get the actual text.
         */
        txt = dataType(topic)==TypeSString ? topic : topic.getTopicText();
        
        /* 
         *   if they don't want an exact case match, convert the original 
         *   topic text to lower case 
         */
        if (!matchExactCase)
            txt = txt.toLower();
        
        /*
         *   To help ensure that we can readily match an appropriate 
         *   matchPattern we strip out any quote marks in which the topic 
         *   text may have been enclosed, and then strip out any duplicate 
         *   spaces.
         */
        
        
        txt = txt.findReplace('"','', ReplaceAll);
        txt = txt.findReplace('  ', ' ', ReplaceAll);   
       
        
        /* 
         *   Also strip out any leading apostrophe-s that might arise from a 
         *   QueryTopic (e.g. ASK HER WHAT'S UP will yield the txt "'s up" 
         *   which the tokenizer will choke on)
         */
        
        if(txt.startsWith('\'s'))
            txt = txt.findReplace('\'s', '', ReplaceOnce);
        
        
        /* 
         *   check for a match to the regular expression pattern, if we have 
         *   a pattern AND the resolved topic allows literal matches 
         */
        if (matchPattern != nil && 
            (dataType(topic)==TypeSString || topic.canMatchLiterally()))
        {
            
            
            /* if the regular expression matches, we match */
            if (rexMatch(matchPattern, txt) != nil) {
                sayTopicManager.addValidCmd(txt);                
                return matchScore;
            }
        }
        
        /*
         *   Finally, see if we can match this topic on its keyword list 
         *   like a SpecialTopic. We can optionally require a minimum number 
         *   of keywords to be matched.
         */
        if(dataType(keywordList) == TypeList)
        {
            local matchCount = 0;
            local toks = Tokenizer.tokenize(txt);
            local vec = new Vector(10);
            local i;
                       
            foreach(local tok in toks)
            {
                tok = getTokVal(tok).toLower();
                                              
                i = keywordList.indexWhich({x: matchKeyword(x, tok, nil)}); 
                
                if(i != nil && vec.indexOf(i) == nil
                   && (enforceKeywordOrder == nil || 
                       vec.indexWhich({x: x > i}) == nil))                
                {
                    matchCount ++;
                    vec.append(i);
                }
                else if(i != nil && allowDuplicateKeywords)
                    continue;
                else if(keywordList.indexWhich({x: matchKeyword(x, tok, true)}) == nil)
                    return nil;            
                
            }
            if(matchCount >= keywordsReqd) {
                sayTopicManager.addValidCmd(txt);                  
                return matchScore;
            } 
        } 
        
        
        /* we didn't find a match - indicate this with a nil score */
        return nil;
    } 
    
    /*
     *   Check whether the given token tok matches the given keyword kwd, 
     *   where kwd may either be a single word or a set of alternative words 
     *   separated by bars (|), e.g. 'came|come'.
     */
    
    matchKeyword(kwd, tok, weak)
    {
        local looksWeak = kwd.startsWith('(') && kwd.endsWith(')');
        
        /* 
         *   First check whether we are meant to be matching a weak or a 
         *   non-weak keyword. If we're the wrong type, we don't match.
         */
        
        if(weak != looksWeak)
            return nil;
        
        /* 
         *   If the bquotes extension is used, the author might put ` 
         *   instead of \' in keywords. Any ` must be changed back to \' 
         *   before attempting to match the player's input against the 
         *   keyword.
         */
        
        kwd = kwd.findReplace('`', '\'', ReplaceAll);
        
        
        /*
         *   If we are a weak keyword, strip off the brackets before 
         *   attempting to make a match. 
         */ 
        if(weak)
            kwd = kwd.substr(2, kwd.length() - 2);
        
        if(kwd == tok)
            return true;
        if(kwd.startsWith(tok + '|') || kwd.endsWith('|' + tok))
            return true;
        if(kwd.find('|' + tok + '|'))
            return true;
        
        return nil;  
    }
    
    
    
    /*
     *   You can define a SpecialTopic-style keywordList which will be tested
     *   for a match if the object match and regular expression matches fail
     *   (or are not defined). You can define a keyword as a weak token by 
     *   enclosing it in parentheses to indicate that it is permitted in a 
     *   response but does not otherwise count towards the match; for 
     *   example, if you're looking for a response like "say Fred is ill", 
     *   you probably don't want to match on "is" alone, so you might define
     *
     *   keywordList = ['fred','(is)', 'ill']
     *
     *   Where alternative phrasings are needed, it may be useful to be able 
     *   to supply alternative forms of non-weak keywords. You can do this 
     *   by separating the alternatives with a | (bar) symbol, e.g. 
     *   'fred|he'. For example, on a QueryTopic that needed to match "ASK 
     *   HIM HOW LONG THE TROUBLES LASTED" and "HOW LONG DID THE TROUBLES 
     *   LAST?" you might define:
     *
     *   keywordList = ['long', '(did)', '(the)', 'troubles', 'last|lasted']
     *
     */
    keywordList = nil
    
    /* 
     *   The minimum number of tokens, excluding weak tokens, for this match 
     *   to succeed. For example, if you want both 'fred' and 'ill' to appear
     *   in a valid response you might set keywordsReqd = 2.    
     *
     *   By default, we set this to the number of non-weak keywords.
     */
    
    keywordsReqd()
    {
        local k;
        
        if(dataType(keywordList==TypeList))
            k = keywordList.countWhich({x: !x.startsWith('(') });
        else  
            k = 0;
        
        /* replace this initializer with the calculated value */
        
        keywordsReqd = k;
        
        return k;
    }
    
    /*
     *   This property determines whether duplicate keywords are allowed in 
     *   a matching response, e.g, "Say fred fred". Normally duplicates will 
     *   result in nonsense, so you want this to be nil. Some phrases may 
     *   need to use the same word twice, however. Note that the duplication 
     *   check is not enforced on weak tokens, so if you define words like 
     *   'a', 'the', 'and' and 'is' as weak keywords, they can be duplicated 
     *   even if more important words aren't.
     */
    allowDuplicateKeywords = nil
    
    /*
     *   You can further enforce the condition that keywords must appear in 
     *   the same order in the player's response as they do in the keyword 
     *   list (again ignoring any weak tokens), so that "fred ill" is 
     *   matched but not "ill fred". Note that settting this to true will 
     *   have no effect if allowDuplicateKeywords is also true. If you need 
     *   both, you're probably better off matching on a regular expression.
     */
    
    enforceKeywordOrder = true
    
    initializeSayTopic()
    {
        if(keywordList == nil)
            return;
        
        /* add each keyword */
        foreach (local cur in keywordList)
        {
            local i;
            if(cur.startsWith('('))
                cur = cur.substr(2);
            if(cur.endsWith(')'))
                cur = cur.substr(1, cur.length() - 1);
            
            while((i = cur.find('|')) != nil)
            {
                cmdDict.addWord(SpecialTopic, cur.substr(1, i-1), &specialTopicWord);    
                cur = cur.substr(i+1);
            } 
            
            /* 
             *   Add the keyword.  Since we don't actually need the 
             *   word-to-object association that the dictionary stores, 
             *   simply associate the word with the SpecialTopic class 
             *   rather than with this particular SayTopic instance. The 
             *   dictionary only stores a given word-obj-prop association 
             *   once, even if it's entered repeatedly, so tying all of the 
             *   special topic keywords to the SpecialTopic class ensures 
             *   that we won't store redundant entries if the same keyword 
             *   is used in multiple say topics.  
             */
            cmdDict.addWord(SpecialTopic, cur, &specialTopicWord);
        }
    }
    
;


sayTopicPreInit: PreinitObject
    execute()
    {
        forEachInstance(SayTopic, { obj: obj.initializeSayTopic() });
    }
;


class DefaultSayTopic: DefaultTopic
    includeInList = [&sayTopics]
    matchScore = 3
;

class DefaultSayTellTopic: DefaultTopic
    includeInList = [&sayTopics, &tellTopics]
    matchScore = 2
;


sayToConvType: ConvType
    topicListProp = &sayTopics
    defaultResponseProp = &defaultSayResponse
    defaultResponse(db, other, topic)
        { db.defaultSayResponse(other, topic); }
;

class SuggestedSayTopic: SuggestedTopic   
    suggestionGroup = [suggestionSayGroup]
;


class QueryTopic: SayTopic
    includeInList = [&queryTopics]
    
    /*
     *   qType can be either a single-quoted string or a list of 
     *   single-quoted strings defining which types of question this 
     *   QueryTopic matches; e.g.  'who' or 'where' or ['if', 'whether']
     *
     *   Note that this word should NOT be repeated in the matchPattern. E.g.
     *   if you want a QueryTopic (or AskQueryTopic, see below) that matches
     *   ASK HER WHO SHE IS then set qType = 'who' and matchPattern = 'she 
     *   is$'
     */
    
    qType = nil
    matchTopic(fromActor, topic)
    {
        /*
         *   If this topic was invoked by a QueryAction, then gAction.qType 
         *   will hold a string defining the type of query (e.g. 'when' or 
         *   'who'). This must match our own qType (or one of our list of 
         *   qTypes) for this topic to be considered a match.
         *
         *   Our subclass AskQueryTopic may also be matched by an AskAction. 
         *   To allow for this possibility we do not enforce the matching of 
         *   qTypes if gAction.qType is nil (as it will be for an AskAction).
         */
        
        
        if(gAction.qType != nil && (
            (dataType(qType)==TypeList && qType.indexOf(gAction.qType.toLower())==nil)
            || (dataType(qType)==TypeSString && qType != gAction.qType.toLower())))
            return nil;
        
        if(topic == nulTopic && sayTopicManager.allowBriefQueries)
            return matchScore;  
        
        
        
        return inherited(fromActor, topic);
    }
    
    
;

class DefaultQueryTopic: DefaultTopic
    includeInList = [&queryTopics]
    matchScore = 3    
    
;

/*
 *   A DefaultAskQueryTopic may be useful for providing default responses to 
 *   all types of question.      
 */

class DefaultAskQueryTopic: DefaultTopic
    includeInList = [&askTopics, &queryTopics]
    matchScore = 2
;

modify DefaultAnyTopic
    includeInList = static inherited + &sayTopics + &queryTopics
;


class SuggestedQueryTopic: SuggestedTopic  
    suggestionGroup = [suggestionQueryGroup]
;

queryConvType: ConvType
    topicListProp = &queryTopics
    defaultResponseProp = &defaultQueryResponse
    defaultResponse(db, other, topic)
        { db.defaultQueryResponse(other, topic); }
;



/*
 *   An AskQueryTopic may be used to match either a general question like 
 *   ASK ABOUT BOB or a more specific one like ASK WHERE BOB WAS BORN in the 
 *   same topic. Use the matchObj to match the target of the ASK ABOUT 
 *   version and a regular expression to match the more specific question.
 */

class AskQueryTopic: QueryTopic
    includeInList = [&askTopics, &queryTopics]
;

class SayTellTopic: SayTopic
    includeInList = [&sayTopics, &tellTopics]
;

modify ActorTopicDatabase
    sayTopics = nil
    queryTopics = nil  
;



modify Thing
    dobjFor(SayTo) {  verify() {illogical(&notAddressableMsg, self); }  }
    dobjFor(Query) {  verify() {illogical(&notAddressableMsg, self); }  }
;

modify Actor
    dobjFor(SayTo)
    {
        preCond = [canTalkToObj]
        verify()
        {
            /* it makes no sense to ask oneself about something */
            verifyNotSelf(&cannotAskSelfMsg);
        }
        action()
        {
            /* note that the issuer is targeting us with conversation */
            gActor.noteConversation(self);
            
            /* let our state object handle it */
            curState.handleConversation(gActor, gTopic, sayToConvType);
        }
    }
    
    dobjFor(Query)
    {
        preCond = [canTalkToObj]
        verify()
        {
            /* it makes no sense to ask oneself about something */
            verifyNotSelf(&cannotAskSelfMsg);
        }
        action()
        {
            /* note that the issuer is targeting us with conversation */
            gActor.noteConversation(self);
            
            /* let our state object handle it */
            curState.handleConversation(gActor, gTopic, queryConvType);
        }
    }
    
    
    defaultSayResponse(fromActor, topic)
        { mainReport(&noResponseFromMsg, self); }
    
    defaultQueryResponse(fromActor, topic)
        { mainReport(&noResponseFromMsg, self); }
    
    obeyCommand(issuingActor, action)
    {
        /* 
         *   We want to trap the situation where the player has issuesd a 
         *   command like BOB, WHERE IS FRED? and turn it into ASK BOB WHERE 
         *   IS FRED
         */
        
        if(issuingActor != self && action.ofKind(QueryAction))
        {
            local cmdStr = sayTopicManager.cmd;
            local i = cmdStr.find(',');
            local actorName;
            if(i)
            {
                actorName = cmdStr.substr(1, i - 1);
                cmdStr = 'ask ' + actorName + ' ' + cmdStr.substr(i + 1);                
            }
    
            local toks = Tokenizer.tokenize(cmdStr);       
        
            executeCommand(gPlayerChar, gPlayerChar, toks, true);
            
            /* 
             *   we've replaced the existing command with the one above, so 
             *   we don't want to continue the command cycle, otherwise this 
             *   will count as two turns.
             */
            throw new TerminateCommandException;
            
        }        
        else
            return inherited(issuingActor, action);       
           
    }
;

/*
 *   Since this extension allows input like WHO ARE YOU, players may add a 
 *   final question-mark, so this StringPreParser removes any terminal 
 *   question-marks.
 */

StringPreParser
    doParsing(str, which)
    {
        if(str.endsWith('?'))
            str = str.findReplace('?','',ReplaceAll);
        sayTopicManager.cmd = str;  
        return str;
    }
    
;

enum SayTopicStrict, SayTopicNormal, SayTopicLax;

sayTopicManager: object
    cmd = nil
    
    /*
     *   mode can be one of SayTopicStrict, SayTopicNormal or SayTopicLax
     */
    
    mode = SayTopicNormal
    validCmds = nil
    maxValidCmds = 30
    
    /*
     *   This routine is called if the parser fails to make sense of player 
     *   input. If mode is not SayTopicStrict and there is a current 
     *   interlocutor we see if the input would match a SayTopic. If it 
     *   would, we display the SayTopic response. If mode = SayTopicNormal 
     *   we ignore aany match   with a DefaultTopic; if mode = SayTopicLax 
     *   we allow a DefaultTopic match.
     *
     *   The routine returns true to indicate that it's dealt with the 
     *   situation, or nil to let the calling routine go on to display its 
     *   standard error message.
     */
    
    tryTopic(actor, txt)
    {
        local resp, cmdStr = cmd, actorName;
        if(mode==SayTopicStrict)
            return nil ;
        if(actor is in (me, nil))
        {
            if ((actor = gPlayerChar.getCurrentInterlocutor()) == nil)
                return errorMsg(actor, txt);
        }
        else
        {
            local i = cmdStr.find(',');
            if(i)
            {
                actorName = cmdStr.substr(1, i - 1);
                cmdStr = cmdStr.substr(i + 1);
            }
        }
        
        local topic = new ResolvedTopicStr(cmdStr); 
        resp = findTopicResponse(actor, topic, sayToConvType); 
        
        if(resp == nil || (mode == SayTopicNormal && resp.ofKind(DefaultTopic)))
            return errorMsg(actor, txt);
        
        // gAction = SayToAction.createInstance();
        // gActor = gPlayerChar;
        // gAction.dobjCur_ = actor;
        
        // gAction.topicList_ = [new ResolveInfo (topic, 0)];
        
        /* 
         *   If there's an actorName, the action was explicitly directed to 
         *   a named actor, e.g. BOB, SUCH AND SUCH, so we want this 
         *   translated to TELL BOB THAT SUCH AND SUCH. Otherwise, the 
         *   statement is directed at the current interlocutor, so we want a 
         *   command of the form SAY SUCH AND SUCH.
         */
        
        if(actorName)        
            cmdStr = 'Tell ' + actorName + ' that ' + cmdStr;
        else
            cmdStr = 'say ' + cmdStr;
        
        local toks = Tokenizer.tokenize(cmdStr);       
        
        executeCommand(gPlayerChar, gPlayerChar, toks, true);
        
        //resp.handleTopic(gActor, gTopic);  
        
        
        addValidCmd(cmd);
        return true;
    }
    
    /*
     *   Find the best match for topic among actor's convType TopicEntries
     */
    
    findTopicResponse(actor, topic, convType)
    {
        local resp;
        local path;
        local node;
        path = [actor.curState, actor]; 
        
        if ((node = actor.curConvNode) != nil)
        {
            resp = node.findTopicResponse(gActor, topic, convType, path);
            if(resp != nil)
                return resp;
        }    
        path = path.sublist(2);
        
        resp = actor.curState.findTopicResponse(gActor, topic, convType, path);
        
        if(resp != nil)
            return resp;
        
        return actor.findTopicResponse(gActor, topic, convType, nil);
        
    }
    
    errorMsg(actor, txt)
    {
        if(validCmds == nil)
            return nil;
        if(validCmds.indexOf(cmd.toLower()))    
            return playerMessages.notValidInThisContext(actor, txt);    
        if(txt == nil)
            return nil;
        
        local t = txt.toLower();
        foreach(local cur in validCmds)
        {
            local toks = Tokenizer.tokenize(cur);
            if(toks.indexWhich({x: getTokVal(x).toLower == t }))
                return playerMessages.notValidOutsideConversation(actor, txt);
            
        }
        return nil;
    }
    
    addValidCmd(str)
    {
        local sl = str.toLower();
        local i;
        if(validCmds == nil)
            validCmds = new Vector(maxValidCmds);
        
        /* 
         *   Check if we already have the command in our list. If so, delete 
         *   the old copy to avoid duplication. We still want the new copy 
         *   added to the start of the list, since it has been used recently.
         */
        
        i = validCmds.indexOf(sl);   
        if(i != nil)  
            validCmds.removeElementAt(i);
        
        
        validCmds.prepend(sl);  
        if(validCmds.length > maxValidCmds)
            validCmds.setLength(maxValidCmds);
    }
    
    /*
     *   If true, will allow commands like ASK WHO or WHERE to match a 
     *   QueryTopic, even though no more of the question is given in the 
     *   command. This is true by default, since this is closer to the 
     *   behaviour of SpecialTopics.
     */
    
    
    allowBriefQueries = true
;

/*
 *   The ResolvedTopicStr class provides a crude way of wrapping a string in 
 *   a ResolvedTopic.
 */

class ResolvedTopicStr: ResolvedTopic
    construct(str)  {   str_ = str;  }
    str_ = nil
    getTopicTokens() { return Tokenizer.tokenize(str_); }
    getTopicText() { return str_; }
    inScopeList = []
    likelyList = []
;

nulTopic: ResolvedTopic
    inScopeList = []
    likelyList = []
;

nulWrapper: object
    obj_ = nulTopic
;


DefineConvTopicTAction(SayTo, DirectObject)
;


DefineConvTopicTAction(Query, IndirectObject)
    /* 
     *   It's possible that the player might word a command in the form BOB, 
     *   HOW ARE YOU? We change this to ASK BOB HOW ARE YOU in 
     *   actor.obeyCommand(), but before that routine the parser will see 
     *   BOB, HOW ARE YOU as a command to BOB to ask HOW ARE YOU, and in 
     *   case of doubt it will ask the play who s/he wants Bob to ask. To 
     *   avoid this redundant question, we always supply the playerChar as 
     *   the default direct object of a Query command when the gActor is not 
     *   the player character.
     */
    getDefaultDobj(np, resolver)
    {
        if(gActor != gPlayerChar)
            return gPlayerChar;
        
        /* get a default direct object using the verify method */
        return inherited(np, resolver);
    }
;

DefineTAction(QueryWhat)  
;

//==================================================================

modify ConvNode
    /*
     *   We'll normally want to show the topic inventory of a ConvNode if it 
     *   contains SayTopics or QueryTopics as well as SpecialTopics
     */
    
    autoShowTopics()
    {
        /*  
         *   if we have an active special topic, query topic, or say topic 
         *   show the topic inventory 
         */
        
        return inherited 
            || (queryTopics != nil
                && queryTopics.indexWhich({x: x.checkIsActive()}) != nil)
            || (sayTopics != nil
                && sayTopics.indexWhich({x: x.checkIsActive()}) != nil);
        
    }
    
;
