////////////////////////////////////////////////////////////////////////
//  
//  ALT Library File: RoomDisplayable 010123
//
//  Copyright (c) 2000, 2001 Kevin Forchione. All rights reserved.
//  Based on ADV.T (c) and STD.T (c) Michael Roberts.
//
//  This file is part of the ALT replacement library for ADV.T and 
//  STD.T and requires TADS 2.5.1 or later.
//
////////////////////////////////////////////////////////////////////////

#ifndef _ROOM_DISPLAYABLE_H_
#define _ROOM_DISPLAYABLE_H_

#include <clock.h>
#include <itemcount.t>
#include <listcont.t>
#include <listcontcont.t>
#include <global.t>

#pragma C+

/*----------------------------------------------------------------------
 *  MIX-IN CLASS:   This class must always preced any non-mix-in classes
 *                  in a superclass list - this is required because a
 *                  mix-in can't override default methods in the root of
 *                  the hierarchy.
 *--------------------------------------------------------------------*/

/*
 *  RoomDisplayable: object
 */
class RoomDisplayable: object
    enterRoom(actor) = {    // sent to room as actor is entering it
        if (self.isLit(actor)) {
            if (!self.isSeen) {
                self.initial;
                "\b";
            }
        }
        self.lookAround((!self.isSeen) || global.verbose);
        if (self.isLit(actor)) {
            if (!self.isSeen) {
                self.dispParagraph;
                self.firstSeen;
            }
            self.isSeen = true;
        }
    }

    leaveRoom(actor) = {
        local tmpList, thisObj, i, tot;
        
        tmpList = self.leaveList;
        tot = length(tmpList);
        i = 1;
        while (i <= tot) {
            thisObj = tmpList[i];
            if (thisObj.leaving(actor))
                self.leaveList = self.leaveList - thisObj;
            i = i + 1;
        }
    }
    /*
     *   Whenever an actor leaves this Room, we run through the leaveList.
     *   This is a list of objects that have registered themselves with us
     *   via addLeaveList().  For each object in the leaveList, we send
     *   a "leaving" message, with the actor as the parameter.  It should
     *   return true if it wants to be removed from the leaveList, nil
     *   if it wants to stay.
     */
    leaveList = []
    addLeaveList(obj) = {
        self.leaveList += obj;
    }
    /*
     *   dispParagraph - display the paragraph separator.  By default we
     *   display a \n\t sequence; games can change this (via 'modify') to
     *   customize the display format 
     */
    dispParagraph = "\n\t"

    /*
     *   dispBeginsDesc and dispEndsDesc - display begin and end of
     *   location short description sequence.  We'll call these just
     *   before and after showing a Room's description.  By default these
     *   do nothing; games that want to customize the display format can
     *   change these (via 'modify').  
     */
    dispBeginsDesc = ""
    dispEndsDesc = ""

    /*
     *   dispBeginlDesc and dispEndlDesc - display begin and end of
     *   location long description sequence.  We'll call these just before
     *   and after showing a Room's long description.  By default, we
     *   start a new paragraph before the lDesc; games that want to
     *   customize the display format can change these (via 'modify').  
     */
    dispBeginlDesc = { self.dispParagraph; }
    dispEndlDesc = ""

    /*
     *   lookAround describes the Room.  If verbosity is true, the full
     *   description is given, otherwise an abbreviated description (without
     *   the Room's lDesc) is displayed.
     */
    lookAround(verbosity) = {
        self.dispBeginsDesc;
        self.statusRoot;
        self.dispEndsDesc;
        
        self.nrmLkAround(verbosity);
    }

    /*
     *  nrmLkAround provides an "immediate" room description
     *  consisting of the location's inside or long description,
     *  special description attributes, and general listing.
     */
    nrmLkAround(verbosity) = {      
        local i, o, len, alsoMess, ret;
        local lookList  = self.contents;

        ////////////////////////////////////////////////////////////////
        //
        //  Add floating items to the contents list
        //
        ////////////////////////////////////////////////////////////////
        
        for (i = 1; i <= length(global.floatingList); ++i) {
            o = global.floatingList[i];
            if (o.location == self
            && find(self.contents, o) == nil)
                lookList += o;
        }
        
        ////////////////////////////////////////////////////////////////
        //
        //  Remove the actor and the actor's location recursively
        //  from the contents list
        //
        ////////////////////////////////////////////////////////////////

        o = parserGetObj(PO_ACTOR);
        if (o == nil) o = parserGetMe();
        while(o) {
            lookList -= [o];
            o = o.location;
        }
                
        if (proptype(self, &describe) != DTY_NIL) {
            self.dispBeginlDesc;
            self.describe;
            self.dispEndlDesc;
        } else if (verbosity) {
            if (proptype(self, &insideDesc) != DTY_NIL) {
                self.dispBeginlDesc;
                self.insideDesc;
                self.dispEndlDesc;
            }
            if (self.location == nil) {
                self.dispBeginlDesc;
                self.lDesc;
                self.dispEndlDesc;
            }
        }
        
        ////////////////////////////////////////////////////////////////
        //
        //  The describe takes precedence over the others.
        //
        ////////////////////////////////////////////////////////////////
        
        len = length(lookList);
        for (i = len; i >= 1; --i) {
            o = lookList[i];
            if (proptype(o, &describe) != DTY_NIL) {
                local ret;
                ret = o.describe;
                if (ret) {
                    lookList -= o;
                    continue;
                }
            }
            
            ////////////////////////////////////////////////////////////
            //
            //  If the object has been moved by the player then we only
            //  display special messages for whenClosed and whenOff.
            //
            ////////////////////////////////////////////////////////////
            if (o.hasMoved) {
                if (o.isContainer
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOff;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                }
            } else {

            /////////////////////////////////////////////////////////////
            //
            //  If the object has not been moved by the player then we
            //  display messages whenOpen, whenClosed, whenOn, whenOff,
            //  and initial.
            //
            /////////////////////////////////////////////////////////////
                if (o.isContainer 
                && o.isOpen
                && proptype(o, &whenOpen) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOpen;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isContainer 
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && o.isActive
                && proptype(o, &whenOn) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOn;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOff;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination 
                && o.isOpen
                && proptype(o, &whenOpen) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOpen;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (proptype(o, &initial) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.initial;
                    self.dispEndlDesc;
                    lookList -= o;                    
                }
            }
        }      

        alsoMess = len - length(lookList);
                  
        self.dispParagraph;
        if (itemCount(lookList)) {
            if (alsoMess)
                "%You% can also see ";
            else
                "%You% see%s% "; 
            listCont(lookList); " here. ";
        }
        listContCont(lookList); "\n";
        
        for (i = 1; i <= length(lookList); ++i) {
            o = lookList[i];
            if (o.isAnimate) {
                self.dispParagraph;
                o.actorDesc;
            }
        }
        "\n";
    }

    /*
     *  xtndLkAround provides a "remote" room description
     *  consisting of the location's inside or long description,
     *  special description attributes, and general listing.
     */
    xtndLkAround(verbosity) = {      
        local i, o, len, alsoMess, ret;
        local lookList  = self.contents;

        ////////////////////////////////////////////////////////////////
        //
        //  Add floating items to the contents list
        //
        ////////////////////////////////////////////////////////////////
        
        for (i = 1; i <= length(global.floatingList); ++i) {
            o = global.floatingList[i];
            if (o.location == self
            && find(self.contents, o) == nil)
                lookList += o;
        }
        
        ////////////////////////////////////////////////////////////////
        //
        //  Remove the actor and the actor's location recursively
        //  from the contents list
        //
        ////////////////////////////////////////////////////////////////

        o = parserGetObj(PO_ACTOR);
        if (o == nil) o = parserGetMe();
        while(o) {
            lookList -= [o];
            o = o.location;
        }
                
        if (proptype(self, &describe) != DTY_NIL) {
            self.dispBeginlDesc;
            self.describe;
            self.dispEndlDesc;
        } else if (verbosity) {
            if (proptype(self, &insideDesc) != DTY_NIL) {
                self.dispBeginlDesc;
                self.insideDesc;
                self.dispEndlDesc;
            }
            if (self.location == nil) {
                self.dispBeginlDesc;
                self.lDesc;
                self.dispEndlDesc;
            }
        }
        
        ////////////////////////////////////////////////////////////////
        //
        //  The describe takes precedence over the others.
        //
        ////////////////////////////////////////////////////////////////
        
        len = length(lookList);
        for (i = len; i >= 1; --i) {
            o = lookList[i];
            if (proptype(o, &describe) != DTY_NIL) {
                local ret;
                ret = o.describe;
                if (ret) {
                    lookList -= o;
                    continue;
                }
            }
            
            ////////////////////////////////////////////////////////////
            //
            //  If the object has been moved by the player then we only
            //  display special messages for whenClosed and whenOff.
            //
            ////////////////////////////////////////////////////////////
            if (o.hasMoved) {
                if (o.isContainer
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL)
                {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOff;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                }
            } else {

            /////////////////////////////////////////////////////////////
            //
            //  If the object has not been moved by the player then we
            //  display messages whenOpen, whenClosed, whenOn, whenOff,
            //  and initial.
            //
            /////////////////////////////////////////////////////////////
                if (o.isContainer 
                && o.isOpen
                && proptype(o, &whenOpen) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOpen;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isContainer 
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && o.isActive
                && proptype(o, &whenOn) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOn;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isSwitchable 
                && !o.isActive
                && proptype(o, &whenOff) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOff;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination
                && o.isOpen
                && proptype(o, &whenOpen) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenOpen;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (o.isDestination
                && !o.isOpen
                && proptype(o, &whenClosed) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.whenClosed;
                    self.dispEndlDesc;
                    lookList -= o;
                } else if (proptype(o, &initial) != DTY_NIL) {
                    self.dispBeginlDesc;
                    o.initial;
                    self.dispEndlDesc;
                    lookList -= o;                    
                }
            }
        }      

        alsoMess = len - length(lookList);
                  
        self.dispParagraph;
        if (itemCount(lookList)) {
            "\^<<self.boardingPrep>> <<self.theDesc>> ";
            if (alsoMess)
                "%you% can also see ";
            else
                "%you% see%s% "; 
            listCont(lookList); " here. ";
        }
        listContCont(lookList); "\n";
        
        for (i = 1; i <= length(lookList); ++i) {
            o = lookList[i];
            if (o.isAnimate) {
                self.dispParagraph;
                o.actorDesc;
            }
        }
        "\n";
    }

    /* 
     *   Generate HTML formatting that provides for the gameClock status 
     *   line appearance and allows us to toggle between a time-oriented
     *   display and the traditional status line appearance, using the 
     *   traditional status line computation mechanism.
     */
#ifdef USE_HTML_STATUS

    statusLine = {
        /*
         *  Check the system to see if this is an HTML-enabled 
         *  run-time. If so, generate an HTML-style banner; otherwise, 
         *  generate an old-style banner. 
         */
        if (systemInfo(__SYSINFO_SYSINFO) == true
        && systemInfo(__SYSINFO_HTML) == 1) {
            self.htmlStatusLine;
        } else {
            /*
             *  Centering the time display around column 40. This
             *  won't be perfect, but it's close enough to the 
             *  center for a comfortable display.
             */
            local i, stat, sr, tsr, len;
            stat = outcapture(true);
            self.statusRoot;
            sr = outcapture(stat);
            stat = outcapture(true);
            self.gameClockStatusRoot;
            tsr = outcapture(stat);
            if (tsr) {
                len = 40 - length(sr) - length(tsr)/2;
            }
            say(sr);
            for (i = 1; i <= len; ++i)
                "\ ";
            say(tsr);
            self.dispParagraph;
        }
    }
    htmlStatusLine = {
 		"<banner id=StatusLine height=previous border>
		<body bgcolor=statusbg text=statustext><b>";

		self.statusRoot;

		"</b>";
		
		if (global.timeStatus == nil || global.timeStatus == 1) {
			"<tab align=center><i>";	
			
            self.gameClockStatusRoot;
            
			"</i>";
		}

		"<tab align=right><i>";
		
		if (global.timeStatus == nil || global.timeStatus == 1) {
			"Score: ";
			say( global.score );
		} else {
            say(scoreFormat(global.score, global.turnSoFar));
            " ";
		}

		"</i></banner>";
    }
    
#else /* USE_HTML_STATUS */

    /* use the standard HTML status line system */
    statusLine = {
        self.statusRoot;
        self.dispParagraph;
    }

#endif /* USE_HTML_STATUS */

    gameClockStatusRoot = {
		if (global.timeStatus == nil) {
			say(gameClock.getClock(CLOCK_DAY_FULL, CLOCK_TIME_HHMMTT));
		} else if (global.timeStatus == 1) {
		    say(gameClock.getClock(CLOCK_DAY_FULL, CLOCK_DATE_EUROPEAN,
		        CLOCK_TIME_HHMMTT));
		}        
    }

    statusRoot = {
        self.sDesc;
    }
    
;

#pragma C-

#endif /* _ROOM_DISPLAYABLE_H_ */
