#charset "us-ascii"

/* ECC: Tads-3 Enterable Complex Container
 * by Steve Breslin & Eric Eve, 2004
 *
 * This module introduces the capability for Actors to enter a complex
 * container.
 *
 */
#include <adv3.h>
#include <en_us.h>

ModuleID
{
    name = 'TADS 3 Enterable Complex Container Library Extension'
    byline = 'by Steve Breslin and Eric Eve'
    htmlByline = 'by <a href="mailto:versim@hotmail.com">Steve
                  Breslin</a> and
                  <a href="mailto:eric.eve@hmc.ox.ac.uk">Eric Eve</a>'
    version = '1.0'
    listingOrder = 72
}

/* ECContainer (Enterable Complex Container) allows an actor to enter
 * behind, under, in or on an ECContainer, provided the appropriate
 * subXXXX property is set to an ECComponent.
 *
 * Note that it's perfectly ok to have one subspace enterable and
 * another not, within the same ECContainer. For example a desk which
 * you can stand on but cannot stand in could be defined thus:
 *
 *. +desk: ECContainer 'desk' 'desk'
 *.    subContainer: ComplexComponent, Container { etc. }
 *.    subSurface: ESubSurface { etc. }
 *. ;
 *
 * ECContainer is not itself a NestedRoom of any kind, just as
 * ComplexContainer is not a Container of any kind: it is designed to
 * remap its "room-like" (and container-like, etc.) behavior to its
 * sub-components. Nevertheless, to maintain the BasicLocation
 * hierarchy, and to allow traversal into and out of its components, it
 * inherits from BasicLocation.
 */
ECContainer: ComplexContainer, BasicLocation

    /* We allow the command without any implied action. Our
     * sub-locations will handle all checks.
     */
    checkMovingActorInto(allowImplicit) { return nil; }
    checkActorOutOfNested(allowImplicit) { return nil; }

    /* We remap verb handling for various "enter"-style commands. */

    dobjFor(StandUnder) maybeRemapTo(subUnderside != nil
                    && subUnderside.ofKind(ECComponent),
                    StandOn, subUnderside)

    dobjFor(GetUnder) maybeRemapTo(subUnderside != nil
                    && subUnderside.ofKind(ECComponent),
                    GetUnder, subUnderside)

    dobjFor(StandBehind) maybeRemapTo(subRear != nil
                    && subRear.ofKind(ECComponent),
                    StandOn, subRear)

    dobjFor(StandOn) maybeRemapTo(subSurface != nil
                    && subSurface.ofKind(ECComponent),
                    StandOn, subSurface)

    dobjFor(StandIn) maybeRemapTo(subContainer != nil
                    && subContainer.ofKind(ECComponent),
                    Board, subContainer)

    dobjFor(SitUnder) maybeRemapTo(subUnderside != nil
                    && subUnderside.ofKind(ECComponent),
                    SitOn, subUnderside)

    dobjFor(SitBehind) maybeRemapTo(subRear != nil
                    && subRear.ofKind(ECComponent),
                    SitBehind, subRear)

    dobjFor(SitOn) maybeRemapTo(subSurface != nil
                    && subSurface.ofKind(ECComponent),
                    SitOn, subSurface)

    dobjFor(SitIn) maybeRemapTo(subContainer != nil
                    && subContainer.ofKind(ECComponent),
                    SitIn, subContainer)

    dobjFor(LieUnder) maybeRemapTo(subUnderside != nil
                    && subUnderside.ofKind(ECComponent),
                    LieOn, subUnderside)

    dobjFor(LieBehind) maybeRemapTo(subRear != nil
                    && subRear.ofKind(ECComponent),
                    LieOn, subRear)

    dobjFor(LieOn) maybeRemapTo(subSurface != nil
                    && subSurface.ofKind(ECComponent),
                    LieOn, subSurface)

    dobjFor(LieIn) maybeRemapTo(subContainer != nil
                    && subContainer.ofKind(ECComponent),
                    LieIn, subContainer)

    dobjFor(Board) asDobjFor(StandIn)

    dobjFor(Enter) asDobjFor(Board)

    /* Likewise, we handle "departure"-style commands. */

    dobjFor(JumpOff) maybeRemapTo(subSurface != nil
                    && subSurface.ofKind(ECComponent),
                    JumpOff, subSurface)

    dobjFor(GetOffOf) asDobjFor(GetOutOf)

    dobjFor(GetOutOf) maybeRemapTo(gActor.isIn(self),
                    GetOutOf, gActor.location)
                    
    nearLoc = (location != nil ? location.roomLocation : nil)
;

/* ECComponent is the base class for ECContainer's components.
 */

class ECComponent: NestedSpace, Component
    exitDestination = (targetObj.nearLoc)
    location = targetObj
    nearLoc = (targetObj.nearLoc)
    dobjFor(StandOn)
    {
        verify()
        {
            if(verifyEntry(standing, &actorAlreadyThereMsg,
                             &noRoomToStandMsg))
                illogical(&cannotStandCloseMsg, position);
        }
        remap() {}
        action() { performEntry(standing); }
    }
    dobjFor(SitOn)
    {
        verify()
        {
            if(verifyEntry(sitting, &actorAlreadyThereMsg,
                             &noRoomToSitMsg))
                illogical(&cannotSitCloseMsg, position);
        }
        remap() {}
        action() { performEntry(sitting); }
    }
    dobjFor(LieOn)
    {
        verify()
        {
            if(verifyEntry(lying, &actorAlreadyThereMsg,
                             &noRoomToLieMsg))
                illogical(&cannotLieCloseMsg, position);
        }
     
    }
    
    /* 
     *   The position property defines how error messages describe attempts to enter
     *   this NestedSpace, e.g. near, under, behind, on. It should be defined as one
     *   of the posXXX values (posNear, posBehind, posUnder, posFront, posIn, posOn).
     *   Subclasses can override this value as appropriate to the kind of space they
     *   represent.
     */
     
    position = posNear
;

class ESubSurface: ECComponent
  position = posOn
;

class ESubContainer: Booth, ECComponent    
    dobjFor(StandOn) { preCond = [touchObj, objOpen] }
    dobjFor(SitIn)
    {
         preCond = [touchObj, objOpen]
         verify()
         {

             /* check whether the actor's already in me  */
             if (gActor.isIn(self) && gActor.posture == sitting)
                 illogicalAlready(&actorPostureThereMsg, self, posIn);

             /* we can't get into something the actor is holding */
             if (isIn(gActor))
                 illogicalNow(&cannotEnterHeldMsg);

             /* Check whether there's enough room for the actor to
              * enter.
               */
             if (!gActor.isIn(self)
                 && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                 illogicalNow(&noRoomInMsg);
         }
         remap() { }
         action()
         {
             performEntry(sitting);
         }
    }

    dobjFor(LieIn)
    {
        preCond = [touchObj, objOpen]
        verify()
        {
 
            /* check whether the actor's already in me */
            if (gActor.isIn(self) && gActor.posture == lying)
                illogicalAlready(&actorPostureThereMsg, self, posIn);


            /* we can't lie in something the actor is holding */
            if (isIn(gActor))
                illogicalNow(&cannotEnterHeldMsg);

            /* Check whether there's enough room for the actor to
             * enter.
             */
            if (!gActor.isIn(self)
                && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                illogicalNow(&noRoomInMsg);

        }

        remap() { }
        action()
        {
            performEntry(lying);
        }
    }
    position = posIn
;

// This definition is for the sake of completeness, but it may not be
// generally useful.
class ESpaceOverlay: SpaceOverlay, ECComponent
;

class ESubUnderside: Underside, ECComponent

    contentsLister = generalContentsLister
    lookInLister = generalContentsLister

    dobjFor(GetUnder)
    {
        preCond = [touchObj]
        verify()
        {
            /* Check whether the actor's already in me.
             */
            if (gActor.isIn(self))
                illogicalAlready(&actorPostureThereMsg, self, posUnder);

            /* we can't get under something the actor is holding */
            if (isIn(gActor))
                illogicalNow(&cannotEnterHeldMsg);

            /* Check whether there's enough room for the actor to
             * enter.
             */
            if (!gActor.isIn(self)
                && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                illogicalNow(&noRoomUnderMsg);
        }

        remap() { }
        action()
        {
            performEntry(defaultPosture);
        }
    }
    
    /* Amend verifyDobjStandOn to give a more meaningful message
     * if the action fails to verify
     */
    position = posUnder
;

class ESubRear: ECComponent, RearContainer
  dobjFor(SitBehind) asDobjFor(SitOn) 
  position = posBehind
;

/* This last class is a bit problematic. Further testing may demonstrate
 * a need to sophisticate this further.
 */
 
// I'm not sure this class is necessary - if a rear space is enterable
// it'll act like a rearContainer, not a rearSurface, won't it? -EE

//-- I think the same logic should apply whether or not the object is
//-- enterable. Perhaps the main problem is we don't have good enough
//-- verbs yet to handle such things as "climb onto the back of the
//-- car".
//--
//-- In any case, this distinction between RearSurface and
//-- RearContainer is made by the main library, and I suppose that's
//-- sufficient reason to maintain the same distinction in this
//-- extension. -SB

class ESubRearSurface: ECComponent, RearSurface
    dobjFor(SitBehind) asDobjFor(SitOn)
    position = posBehind
;

