#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>
#include "dynfunc.h"

// Version history 
// 0.21 2019-05-20 before rewrite to prevent dependency on dynfunc.t
// 0.30 2019-05-22 after rewrite to prevent dependency on dynfunc.t

// -----------------------------------------------------------------------------
// Table of contents
// -----------------------------------------------------------------------------

// Instructions
// Game basics
// Overriding the achievement-scope filename
// Achievement examples
// Examples of objects whose properties we can manipulate
// endGame function
// Verbs/actions: Achievement-scope variables: AGET (read value from achievement-scope object)
// Verbs/actions: Achievement-scope variables: ASET (store value to achievement-scope object)
// Verbs/actions: Achievement-scope list variables: ALISTCLEAR (set an achievement-scope list to empty)
// Verbs/actions: Achievement-scope list variables: ALISTADD (add a value to an achievement-scope list)
// Verbs/actions: Achievement-scope list variables: ALISTREMOVE (remove a value from an achievement-scope list)
// Verbs/actions: Data translation/storage: ASAVE1 (Achievement data from objects to memory)
// Verbs/actions: Data translation/storage: ASAVE2 (Achievement data encryption)
// Verbs/actions: Data translation/storage: ASAVE3 (Achievement data from objects to memory to disk)
// Verbs/actions: Data translation/storage: ALOAD1 (Achievement data from disk to memory)
// Verbs/actions: Data translation/storage: ALOAD2 (Achievement data from disk to memory to objects)
// Verbs/actions: Achievements: ACHIEVE (magically achieve an achievement)
// Verbs/actions: Achievements: UNACHIEVE (magically unachieve an achievement)
// Verbs/actions: Achievements: ERASE ALL (magically delete all achievement-scope data!)
// Verbs/actions: Achievements: LOSE (magically lose the game)
// Verbs/actions: Achievements: WIN (magically win the game)
// Verbs/actions: Achievements: ASDF
// Verbs/actions: Achievements: REVEAL (magically reveal an achievement)
// Verbs/actions: Achievements: CONCEAL (magically conceal an achievement)

// -------------------------------------------------------------------------------------
// Instructions
// -------------------------------------------------------------------------------------

// To learn how to develop a game that has achievements using achievement.t:
// 1. Read the instructions at the beginning of achievement.t
// 2. Try compiling a game using reflect.t, dynfunc.t, achievement.t, and achievementdemo.t;
//    then use commands like in the following walkthroughs to get a sense of how to use
//    this demo. 

// WALKTHROUGH OF ACHIEVEMENT FUNCTIONALITY
// 
// Command                 Code                                              Comments
// -------                 ----                                              --------
// achievements            [achievement.t standard action]                   0 of 5 achievements attained.
// lose                    achLoser.incrementCount(); finishGameMsg(...);    "You have failed"
// undo                    [TADS 3 standard action]                          Undo last action.
// lose                    achLoser.incrementCount(); finishGameMsg(...);    "You have failed"
// undo                    [TADS 3 standard action]                          Undo last action.
// lose                    achLoser.incrementCount(); finishGameMsg(...);    "You have failed". Unlock 1st achievement "Loser". Get one-time advisory "to see the list of achievements..."
// achievements            [achievement.t standard finish type option]       1 of 5 achievements attained.
// undo                    [TADS 3 standard action]                          Undo last action.
// achievements            [achievement.t standard action]                   1 of 5 achievements attained. (not undone by "undo")
// win                     achWinner.achieve(); finishGameMsg(...);          "You have won". Unlock 2nd achievement: "Winner". [You should not see the one-time advisory.]
// undo                                                                      
// achievements            [achievement.t standard action]                   2 of 5 achievements attained.
// achieve achAchiever     achAchiever.achieve();                            Unlock 3rd achievement: "Achiever". 
// unachieve achAchiever   achUnachiever.achieve();                          Unlock 4rd achievement: "Unachiever." 
// achieve achAchiever     achAchiever.achieve();                            
// achievements            [achievement.t standard action]                   4 of 5 achievements attained. 
// asdf                    achAsdf.reveal();                                 
// achievements            [achievement.t standard action]                   The last achievement is now revealed but not unlocked. 
// asdf                    asdf                                              
// asdf                                                                      
// asdf                    achAsdf.achieve();                                Unlock 5th achievement: Asdf.
// achievements            [achievement.t standard action]                   5 of 5 achievements attained. 
// QUIT WITHOUT SAVING                                                       
// CLOSE INTERPRETER                                                         
// OPEN INTERPRETER                                                          
// START NEW GAME                                                            
// achievements            [achievement.t standard action]                   5 of 5 achievements attained. 

// WALKTHROUGH OF ACHIEVEMENT-SCOPE VARIABLE FUNCTIONALITY
// 
// Command                 Code                                              Desired result              Comments
// -------                 ----                                              --------------              --------
// aget avar               avar.getValue(); (or just avar.value)             avar = 3                    Get value from achievement-scope variable (non-typed)
// aget abool              abool.getValue(); (or just abool.value)           abool = nil                 Get value from achievement-scope variable (boolean)
// aget aint               aint.getValue(); (or just aint.value)             aint = 4                    Get value from achievement-scope variable (integer)
// aget astr               astr.getValue(); (or just astr.value)             astr = 'asdf'               Get value from achievement-scope variable (string)
// aget aboollist          aboollist.getValue(); (or just aboollist.value)   aboollist = [nil]           Get value from achievement-scope list variable (boolean)
// aget aintlist           aintlist.getValue(); (or just aintlist.value)     aintlist = [3,4,5]          Get value from achievement-scope list variable (integer)
// aget astrlist           astrlist.getValue(); (or just astrlist.value)     astrlist = ['asdf','qwer']  Get value from achievement-scope list variable (string)
// aset avar 103           avar.setValue('103');                             Okay, avar = '103'          Set value of achievement-scope variable (non-typed)
// aset abool true         abool.setValue(true);                             Okay, abool = true          Set value of achievement-scope variable (boolean)
// aset aint 104           aint.setValue(104);                               Okay, aint = 104            Set value of achievement-scope variable (integer)
// aset astr zxcv          astr.setValue('zxcv');                            Okay, astr = 'zxcv'         Set value of achievement-scope variable (string)
// alistclear aboollist    aboollist.clearList();                            Okay, aboollist = []        Clear list for achievement-scope list variable (boolean)
// alistclear aintlist     aintlist.clearList();                             Okay, aintlist = []         Clear list for achievement-scope list variable (int)
// alistclear astrlist     astrlist.clearList();                             Okay, astrlist = []         Clear list for achievement-scope list variable (string)
// alistadd aboollist true aboollist.addValue(true);                         Okay, aboollist = [true]    Add value to list for achievement-scope list variable (boolean)
// alistadd aintlist 103   aintlist.addValue(103);                           Okay, aintlist = [103]      Add value to list for achievement-scope list variable (integer)
// alistadd astrlist zxcv  astrlist.addValue('zxcv');                        Okay, astrlist = ['zxcv']   Add value to list for achievement-scope list variable (string)
// QUIT WITHOUT SAVING                                                                                   
// CLOSE INTERPRETER                                                                                     
// OPEN INTERPRETER                                                                                      
// START NEW GAME                                                                                        
// aget avar               avar.getValue(); (or just avar.value)             avar = '103'                Retains the value that we set above
// aget abool              abool.getValue(); (or just abool.value)           abool = true                Retains the value that we set above
// aget aint               aint.getValue(); (or just aint.value)             aint = 104                  Retains the value that we set above
// aget astr               astr.getValue(); (or just astr.value)             astr = 'zxcv'               Retains the value that we set above
// aget aboollist          aboollist.getValue(); (or just aboollist.value)   aboollist = [true]          Retains the value that we set above
// aget aintlist           aintlist.getValue(); (or just aintlist.value)     aintlist = [103]            Retains the value that we set above
// aget astrlist           astrlist.getValue(); (or just astrlist.value)     astrlist = ['zxcv']         Retains the value that we set above
// asave1                  [Authors need not call methods called by Asave1]  See comments -->            Gets achievement-scope data from objects, translates to file-friendly format, and stores to memory
// asave2                  [Authors need not call methods called by Asave2]  See comments -->            Shows successful encrypted text
// asave3                  [Authors need not call methods called by Asave3]  See comments -->            Gets achievement-scope data from memory, and stores to file
// aload1                  [Authors need not call methods called by Aload1]  See comments -->            Gets achievement-scope data from file, and stores to memory
// aload2                  [Authors need not call methods called by Aload2]  See comments -->            Gets achievement-scope data from memory, translates, and stores to objects

// -------------------------------------------------------------------------------------
// Game basics
// -------------------------------------------------------------------------------------

versionInfo: GameID
  IFID = '23e896b9-a830-47e1-926e-44a79d9acc3b'
  name = 'Achievement Simple Demo'
  byline = 'by Greg Boettcher'
  htmlByline = 'by <a href="mailto:paradoxgreg@gmail.com">Greg Boettcher</a>'
  version = '1.00'
  authorEmail = 'Greg Boettcher <paradoxgreg@gmail.com>'
  desc = 'Demonstrates the use of the TADS 3 module achievement.t.'
  htmlDesc = 'Demonstrates the use of the TADS 3 module achievement.t.'
;

gameMain: GameMainDef
  initialPlayerChar = me
  showIntro() {
    "Read the comments at the beginning of achievementdemo.t 
        for details about the commands you can use in this demo to test 
        achievement.t. \b";
  }
;

me: Actor
  location = entrance
;

entrance: OutdoorRoom 'Entrance' 'the entrance' 'entrance'
  "There's no way out of here. "
;

// -----------------------------------------------------------------------------
// Overriding the achievement-scope filename
// -----------------------------------------------------------------------------

// Overriding the achievement-scope filename is good practice, as it prevents
// problems in the unlikely event that your .t3 game puts in the same folder 
// as another .t3 game that also uses achievement.t. 

modify achievementScopeFileManager
  achievementFilename = 'achievementdemo.sav'
;

// -----------------------------------------------------------------------------
// Achievement examples
// -----------------------------------------------------------------------------

transient achAchiever: ASAchievement
  asName = 'achAchiever'
  name = 'Achiever'
  desc = 'Achieve this achievement by typing "ACHIEVE achAchiever" (case-sensitive) '
  listOrder = 10
;

transient achUnachiever: ASAchievement
  asName = 'achUnachiever'
  name = 'Unachiever'
  desc = 'Unachieve any achievement by typing "UNACHIEVE [nameOfAchievementObject]" (case-sensitive) '
  listOrder = 20
  visible = nil // not initially visible by name
;

transient achAsdf: ASAchievement
  asName = 'achAsdf'
  name = 'Asdf'
  desc = 'Achieve this achievement by typing "ASDF" 5 times in a single playthrough '
  listOrder = 30
  visible = nil // not initially visible by name
;

transient achLoser: ASAchievement
  asName = 'achLoser'
  name = 'Loser'
  desc = 'Lose the game three times by typing "LOSE" '
  listOrder = 40
  incrementCount() {
    // Storing the count to an achievement-scope variable; see comments below
    achLoserCount.setValue(achLoserCount.getValue() + 1);
    if (achLoserCount.getValue() >= 3) {
      achieve();
    }
  }
;

// If we wanted to, we could implement this as a property of achLoser; we would just
// need to add the property to that object and then override saveablePropertyInfo. 
// Either way the property needs to be added to either an ASAchievement or an ASVariable 
// that is transient and whose saveablePropertyInfo includes that property. 
transient achLoserCount: ASInteger
  asName = 'achLoserCount'
  value = 0
;

transient achWinner: ASAchievement
  asName = 'achWinner'
  name = 'Winner'
  desc = 'Win the game by typing "WIN" '
  listOrder = 50
;

// -----------------------------------------------------------------------------
// Examples of objects whose properties we can manipulate
// -----------------------------------------------------------------------------

// All of the objects here are transient objects, because they all NEED
// to be transient objects in order to have the desired functionality of 
// (SET VALUE = 2, UNDO, GET VALUE resulting in the number 2). 

transient avar: ASVariable
  asName = 'avar'
  // It's okay to override "value" to set an initial value, but after this 
  // we'll need to use setValue() if we want to change the value.
  value = 3
;

transient aint: ASInteger
  asName = 'aint'
  value = 4
;

transient abool: ASBoolean
  asName = 'abool'
  value = nil
;

transient astr: ASString
  asName = 'astr'
  value = 'asdf'
;

//transient ASString
//  value = 'I\'m an anonymous object. Therefore, achievement.t won\'t try to store data to me. '
//;

transient avarlist: ASList
  asName = 'avarlist'
  value = [] // Some of my tests assume this to be an empty list
;

transient astrlist: ASStringList
  asName = 'astrlist'
  value = ['asdf','qwer']
;

transient aintlist: ASIntegerList
  asName = 'aintlist'
  value = [3,3,4,4,5,5]
;

transient aboollist: ASBooleanList
  asName = 'aboollist'
  value = [nil]
;

// tobj:
// a plain old object, except it's transient (thus retaining its value even if you, 
// e.g., type UNDO after doing the thing that changed its value) 

transient tobj: object
  value = 1
  value2 = 2
  // objprop is a property of transient object tobj. this property is valued
  // as an anonymous object that is not transient. 
  objprop : object {
    value = 123
    objprop : object {
      value = 456
    }
  }
;

class SuggestedTransient: object
  value = 123
  //// This isn't valid syntax:
  //objprop : transient object {
  //  value = 123
  //}
  // This is valid syntax:
  objprop : object {
    value = 123
  }
;

transient ost: SuggestedTransient
  //// This isn't valid syntax:
  //objprop2 : transient object {
  //  value = 123
  //}
  //// This isn't valid syntax:
  //transient objprop2 : object {
  //  value = 123
  //}
  // This is valid syntax:
  objprop2 : object {
    value = 123
  }
;

// -----------------------------------------------------------------------------
// endGame function
// -----------------------------------------------------------------------------

// This endGame function brought to you by Eric Eve, slightly modified 
// from his Getting Started in TADS 3. 

function endGame(ft) {
  if (ft == ftFailure || ft == ftDeath || ft == ftGameOver) {
    // Every time you lose the game, increment an achievement-scope variable
    // to keep track of how many times you've done so.
    achLoser.incrementCount();
  }
  if (ft == ftVictory) {
    achWinner.achieve();
  }
  
  finishGameMsg(ft, [finishOptionUndo, finishOptionAchievements]);
}

//// -----------------------------------------------------------------------------
//// Verbs/actions: Standard objects: TGET (read value from objects)
//// -----------------------------------------------------------------------------
//
//VerbRule(Tget)
//  'tget' singleLiteral
//  : TgetAction
//  verbPhrase = 'tget/tgetting (what)'
//;
//
//DefineLiteralAction(Tget)
//  execAction() {
//    // Parse input
//    // example: "tget a b c"
//    local str = getLiteral();
//    local tokens = str.split(' ');
//    if (tokens.length < 1) {
//      sayExampleUsage();
//      return;
//    }
//    // For the above example, obtain the string 'a.b.c'
//    local parseName = '';
//    for (local i = 1; i <= tokens.length; i++) {
//      if (i > 1) { parseName += '.'; }
//      parseName += tokens[i];
//    }
//    
//    // Get the value of Compiler.eval('a.b.c'). 
//    local valueOfParseName;
//    try {
//      valueOfParseName = Compiler.eval(parseName);
//    }
//    catch (Exception ex) {
//      "Error in trying to get value of <<parseName>>. 
//      <p>Exception message: <<ex.errmsg_>>\n";
//      return;
//    }
//    
//    // If the value of 'a.b.c' was obtainable and was found to be anything other 
//    // than an object, say the value and declare victory. 
//    if (dataType(valueOfParseName) != TypeObject) {
//      "Value of <<parseName>>: ";
//      "\n<<valueOfParseName>> ";
//      return;
//    }
//    
//    // If the value of, e.g., 'a.b.c' was obtainable and was found to be an object, 
//    // evaluate 'a.b.c.value' rather than 'a.b.c'. If that works, say the value
//    // of 'a.b.c.value' and declare victory. If it doesn't, give an error message. 
//    try {
//      parseName = parseName + '.value';
//      valueOfParseName = Compiler.eval(parseName);
//      
//      "Value of <<parseName>>: ";
//      "\n<<valueOfParseName>> ";
//    }
//    catch (Exception ex) {
//      "Error in trying to get value of <<parseName>>. 
//      <p>Exception message: <<ex.errmsg_>>\n";
//    }
//  }
//  sayExampleUsage() {
//    "TGET: Retrieve the value of a property of an object within an object. 
//      <p>Example usage, where prop1 is a property of object obj, and prop1 is valued as a string or integer:
//      \nTGET obj prop1
//      <p>Example usage, where prop1 is a property of object obj, and prop1 is valued as an object with a value property that is valued as a string or integer:
//      \nTGET obj prop1
//      \nTGET obj prop1 value ";
//  }
//  actionTime = 0
//;
//
//// To give a better response to just 'TGET'
//
//DefineIAction(TgetIntransitive)
//  execAction() {
//    TgetAction.sayExampleUsage();
//  }
//  actionTime = 0
//;
//
//VerbRule(TgetIntransitive)
//  'tget'
//  : TgetIntransitiveAction
//  verbPhrase = 'tget/tgetting'
//;
//
//// -----------------------------------------------------------------------------
//// Verbs/actions: Standard objects: TSET (write value to objects)
//// -----------------------------------------------------------------------------
//
//// As implemented here, this always sets the value to be a string.
//
//// Note: This action is extremely powerful and could be used to damage 
//// the inner mechanics of TADS 3. Do not put this code into an actual game.
//
//VerbRule(Tset)
//  'tset' singleLiteral
//  : TsetAction
//  verbPhrase = 'tset/tsetting (what)'
//;
//
//DefineLiteralAction(Tset)
//  execAction() {
//    // Parse input
//    // example: "tget a b c 5"
//    local str = getLiteral();
//    local tokens = str.split(' ');
//    if (tokens.length < 2) {
//      sayExampleUsage();
//      return;
//    }
//    // For the above example, obtain curParseName = 'a.b.c', and prevParseName = 'a.b'
//    local prevParseName = '';
//    local curParseName = '';
//    local newValue = tokens[tokens.length];
//    for (local i = 1; i <= tokens.length - 1; i++) {
//      prevParseName = curParseName;
//      
//      if (i > 1) { curParseName += '.'; }
//      curParseName += tokens[i];
//    }
//    
//    // Try to parse, e.g. for the example above, 'a.b.c' as an object. 
//    local obj;
//    local prop;
//    local parseName = '';
//    local propName = '';
//    local successfullyParsed = nil;
//    try {
//      obj = Compiler.eval(curParseName);
//      if (dataType(obj) == TypeObject) {
//        propName = '&value';
//        parseName = curParseName;
//        prop = Compiler.eval(propName);
//        successfullyParsed = true;
//      }
//    }
//    catch (Exception ex) {
//      "Error in trying to parse object/property: <<curParseName>>/<<propName>>. 
//      <p>Exception message: <<ex.errmsg_>>\n";
//      return;
//    }
//    
//    // In the example above, if 'a.b.c' cannot be parsed as an object, 
//    // try to parse 'a.b' as an object (with 'c' as its property).
//    if (!successfullyParsed) {
//      try {
//        obj = Compiler.eval(prevParseName);
//        if (dataType(obj) == TypeObject) {
//          parseName = prevParseName;
//          propName = '&' + tokens[tokens.length - 1];
//          prop = Compiler.eval(propName);
//          successfullyParsed = true;
//        }
//      }
//      catch (Exception ex) {
//        "Error in trying to parse object/property: <<curParseName>>/<<propName>>. 
//        <p>Exception message: <<ex.errmsg_>>\n";
//        return;
//      }
//    }
//    
//    // If no object found, say so and give up
//    if (!successfullyParsed) {
//      if (prevParseName == curParseName) {
//        "Error: <<curParseName>> is not interpretable as an object. ";
//      }
//      else {
//        "Error: neither <<prevParseName>> nor <<curParseName>> are 
//          interpretable as objects. ";
//      }
//      return;
//    }
//    
//    // Set the value
//    try {
//      obj.(prop) = newValue;
//      "Okay, <<parseName>>.<<propName>> has been set to <<newValue>>. ";
//    }
//    catch (Exception ex) {
//      "Error in trying to set value of <<parseName>>.<<propName>> to <<newValue>>. 
//      <p>Exception message: <<ex.errmsg_>>\n";
//    }
//  }
//  sayExampleUsage {
//    "TSET: Store a value to an object property. (Use this for good, not for evil--!)
//      <p>Example usage:\n
//      TSET objectName newValue\n
//      TSET objectName propertyName newValue ";
//  }
//  actionTime = 0
//;
//
//// To give a better response to just 'TSET'
//
//DefineIAction(TsetIntransitive)
//  execAction() {
//    TsetAction.sayExampleUsage();
//  }
//  actionTime = 0
//;
//
//VerbRule(TsetIntransitive)
//  'tset'
//  : TsetIntransitiveAction
//  verbPhrase = 'tset/tsetting'
//;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievement-scope variables: AGET (read value from achievement-scope object)
// -----------------------------------------------------------------------------

VerbRule(Aget)
  'aget' singleLiteral
  : AgetAction
  verbPhrase = 'aget/agetting (what)'
;

DefineLiteralAction(Aget)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1 ) {
      sayExampleUsage();
      return;
    }
    
    local objectName = tokens[1];
    try {
      //local asObj = Compiler.eval(objectName);
      local asObj = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (asObj == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      local asValue = asObj.getValueForDebugging();
      //local asValue = toString(asObj.getValue());
      "Value of achievement-scope variable <<objectName>>: ";
      "\n<<asValue>>&nbsp;";
    }
    catch (Exception ex) {
      "Error in trying to get achievement-scope value 
      <<objectName>>.getValueForDebugging(): 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  actionTime = 0
  sayExampleUsage() {
    "AGET: Retrieve the value of an achievement-scope variable.
      <p>Example usage:\n
      AGET achievementScopeVariableName";
  }
;

// To give a better response to just 'AGET'

DefineIAction(AgetIntransitive)
  execAction() {
    AgetAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AgetIntransitive)
  'aget'
  : AgetIntransitiveAction
  verbPhrase = 'aget/agetting'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievement-scope variables: ASET (store value to achievement-scope object)
// -----------------------------------------------------------------------------

VerbRule(Aset)
  'aset' singleLiteral
  : AsetAction
  verbPhrase = 'aset/asetting (what)'
;

DefineLiteralAction(Aset)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 2) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    local newValue = tokens[2];
    
    try {
      //local asObj = Compiler.eval(objectName);
      local asObj = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (asObj == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      asObj.setValue(newValue);
      "Okay, the value <<newValue>> has been added to 
        achievement-scope variable <<objectName>>.  
        <p>New value of variable: 
        \n<<toString(asObj.getValueForDebugging())>>. ";
    }
    catch (Exception ex) {
      "Error in trying to set achievement-scope variable <<objectName>> to <<newValue>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "ASET: Store a value to an achievement-scope variable. 
      <p>Example usage:\n
      ASET achievementScopeVariableName newValue ";
  }
  actionTime = 0
;

// To give a better response to just 'ASET'

DefineIAction(AsetIntransitive)
  execAction() {
    AsetAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AsetIntransitive)
  'aset'
  : AsetIntransitiveAction
  verbPhrase = 'aset/asetting'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievement-scope list variables: ALISTCLEAR (set an achievement-scope list to empty)
// -----------------------------------------------------------------------------

VerbRule(Alistclear)
  'alistclear' singleLiteral
  : AlistclearAction
  verbPhrase = 'alistclear/alistclearing (what)'
;

DefineLiteralAction(Alistclear)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    
    try {
      //local asObj = Compiler.eval(objectName);
      local asObj = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (asObj == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      asObj.clearList();
      "Okay, the achievement-scope list <<objectName>>
        has been cleared out. 
        <p>New value of list: 
        \n<<toString(asObj.getValueForDebugging())>> ";
    }
    catch (Exception ex) {
      "Error in trying to clear the achievement-scope list <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "ALISTCLEAR: Set an achievement-scope list to be an empty list. 
      <p>Example usage:
        \nALISTCLEAR achievementScopeList";
  }
  actionTime = 0
;

// To give a better response to just 'ALISTCLEAR'

DefineIAction(AlistclearIntransitive)
  execAction() {
    AlistclearAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AlistclearIntransitive)
  'alistclear'
  : AlistclearIntransitiveAction
  verbPhrase = 'alistclear/alistclearing'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievement-scope list variables: ALISTADD (add a value to an achievement-scope list)
// -----------------------------------------------------------------------------

VerbRule(Alistadd)
  'alistadd' singleLiteral
  : AlistaddAction
  verbPhrase = 'alistadd/alistadding (what)'
;

DefineLiteralAction(Alistadd)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 2) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    local newValue = tokens[2];
    
    try {
      //local asObj = Compiler.eval(objectName);
      local asObj = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (asObj == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      asObj.addValue(newValue);
      "Okay, the value <<newValue>> has been added to achievement-scope list <<objectName>>. 
        <p>New value of list: 
        \n<<toString(asObj.getValueForDebugging())>> ";
    }
    catch (Exception ex) {
      "Error in trying to add value <<newValue>> to achievement-scope list <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "ALISTADD: Add a value to an achievement-scope list.
      <p>Example usage:
        \nALISTADD achievementScopeList valueToAdd ";
  }
  actionTime = 0
;

// To give a better response to just 'ALISTADD'

DefineIAction(AlistaddIntransitive)
  execAction() {
    AlistaddAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AlistaddIntransitive)
  'alistadd'
  : AlistaddIntransitiveAction
  verbPhrase = 'alistadd/alistadding'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievement-scope list variables: ALISTREMOVE (remove a value from an achievement-scope list)
// -----------------------------------------------------------------------------

VerbRule(AlistRemove)
  'alistremove' singleLiteral
  : AlistremoveAction
  verbPhrase = 'alistremove/alistremoving (what)'
;

DefineLiteralAction(Alistremove)
  execAction() {
    // Parse input: object name + optional "ALL" keyword
    local replaceAllInstances = nil;
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length < 2 || tokens.length > 3) {
      sayExampleUsage;
      return;
    }
    if (tokens.length == 3) {
      if (tokens[3].toUpper() != 'ALL') {
        sayExampleUsage;
        return;
      }
      replaceAllInstances = true;
    }
    local objectName = tokens[1];
    local newValue = tokens[2];
    
    // Do the removal from the list
    try {
      //local asObj = Compiler.eval(objectName);
      local asObj = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (asObj == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      if (replaceAllInstances) {
        asObj.removeValue(newValue, true);
      }
      else {
        asObj.removeValue(newValue);
      }
      "Okay, the value <<newValue>> has been removed from achievement-scope list <<objectName>>. 
        <p>New value of list: 
        \n<<toString(asObj.getValueForDebugging())>> ";
    }
    catch (Exception ex) {
      "Error in trying to remove value <<newValue>> from achievement-scope list <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "ALISTREMOVE: Remove a value from an achievement-scope list.
      <p>Example usage:
        \nALISTREMOVE achievementScopeList valueToRemove 
        \nALISTREMOVE achievementScopeList valueToRemove ALL";
  }
  actionTime = 0
;

// To give a better response to just 'ALISTREMOVE'

DefineIAction(AlistremoveIntransitive)
  execAction() {
    AlistremoveAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AlistremoveIntransitive)
  'alistremove'
  : AlistremoveIntransitiveAction
  verbPhrase = 'alistremove/alistremoving'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Data translation/storage: ASAVE1 (Achievement data from objects to memory)
// -----------------------------------------------------------------------------

// ASAVE1: Test the collection of data from achievement-scope objects and the storing
// of them to memory in a format that be saved to disk.

DefineIAction(Asave1)
  execAction() {
    //achivementScopeManager.saveAllAchievementScopeData();
    achievementScopeTranslator.getAchievementScopeDataFromObjects();
    say(achievementScopeTranslator.data);
  }
  actionTime = 0
;

VerbRule(Asave1)
  'asave1'
  : Asave1Action
  verbPhrase = 'asave/asaving Step 1'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Data translation/storage: ASAVE2 (Achievement data encryption)
// -----------------------------------------------------------------------------

// ASAVE2: Test the encryption of achievement-scope data.

DefineIAction(Asave2)
  execAction() {
    if (achievementScopeTranslator.data == nil ||
        achievementScopeTranslator.data == '' ||
        achievementScopeTranslator.data == [])
    {
      achievementScopeTranslator.getAchievementScopeDataFromObjects();
    }
    if (listBasedAchievementScopeDataTranslator.data == nil ||
        listBasedAchievementScopeDataTranslator.data == '' ||
        listBasedAchievementScopeDataTranslator.data == [])
    {
      listBasedAchievementScopeDataTranslator.getAchievementScopeDataFromObjects();
    }
    
    "Encryption of achievementScopeTranslator.data:\n";
    say(achievementScopeEncrypter.encryptString(achievementScopeTranslator.data));
    
    "<p>";
    "Encryption of listBasedAchievementScopeDataTranslator.data:\n";
    for (local i = 1; i <= listBasedAchievementScopeDataTranslator.data.length; i++) {
      say(achievementScopeEncrypter.encryptString(listBasedAchievementScopeDataTranslator.data[i]) + '\n');
    }
  }
  actionTime = 0
;

VerbRule(Asave2)
  'asave2'
  : Asave2Action
  verbPhrase = 'asave/asaving Step 2'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Data translation/storage: ASAVE3 (Achievement data from objects to memory to disk)
// -----------------------------------------------------------------------------

// ASAVE3: Test the whole process of collecting achievement-scope data from objects
// and saving them to disk.

DefineIAction(Asave3)
  execAction() {
    // Step 1:
    if (achievementScopeTranslator.data == nil ||
        achievementScopeTranslator.data == '' ||
        achievementScopeTranslator.data == [])
    {
      achievementScopeTranslator.getAchievementScopeDataFromObjects();
    }
    if (listBasedAchievementScopeDataTranslator.data == nil ||
        listBasedAchievementScopeDataTranslator.data == '' ||
        listBasedAchievementScopeDataTranslator.data == [])
    {
      listBasedAchievementScopeDataTranslator.getAchievementScopeDataFromObjects();
    }
    
    // Steps 2 and 3:
    achievementScopeFileManager.saveCollectedDataToFile();
    achievementScopeFileManager.saveListBasedCollectedDataToFile();
    
    "Stored to two files. ";
  }
  actionTime = 0
;

VerbRule(Asave3)
  'asave3'
  : Asave3Action
  verbPhrase = 'asave/asaving Step 3'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Data translation/storage: ALOAD1 (Achievement data from disk to memory)
// -----------------------------------------------------------------------------

// ALOAD1: Test the retrieval of data from achievement-scope objects and the storing
// of them to memory in a format that be saved to disk.

DefineIAction(Aload1)
  execAction() {
    achievementScopeTranslator.clearData();
    listBasedAchievementScopeDataTranslator.clearData();
    // Put in some bogus data we don't want to see
    achievementScopeTranslator.addToData('avarlist\tvalue\tlst\t8\tbogus data 1');
    achievementScopeTranslator.addToData('avarlist\tvalue\tlst\t8\tbogus data 2');
    listBasedAchievementScopeDataTranslator.addToData('avarlist\tvalue\tlst\t8\tbogus data 1');
    listBasedAchievementScopeDataTranslator.addToData('avarlist\tvalue\tlst\t8\tbogus data 2');
    
    "BEFORE:";
    "<p>";
    "achievementScopeTranslator.data:\n";
    say(achievementScopeTranslator.data);
    "<p>";
    "listBasedAchievementScopeDataTranslator.data:\n";
    for (local i = 1; i <= listBasedAchievementScopeDataTranslator.data.length; i++) {
      say(listBasedAchievementScopeDataTranslator.data[i] + '\n');
    }
    "<p>";
    
    achievementScopeFileManager.loadDataFromFileToMemory();
    achievementScopeFileManager.loadListBasedDataFromFileToMemory();
    
    "AFTER:";
    "<p>";
    "achievementScopeTranslator.data:\n";
    say(achievementScopeTranslator.data);
    "<p>";
    "listBasedAchievementScopeDataTranslator.data:\n";
    for (local i = 1; i <= listBasedAchievementScopeDataTranslator.data.length; i++) {
      say(listBasedAchievementScopeDataTranslator.data[i] + '\n');
    }
    "\n";
  }
  actionTime = 0
;

VerbRule(Aload1)
  'aload1'
  : Aload1Action
  verbPhrase = 'aload/aloading Step 1'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Data translation/storage: ALOAD2 (Achievement data from disk to memory to objects)
// -----------------------------------------------------------------------------

// ALOAD2: Test the whole process of reading from the achievement save file (just the one)
// and storing the values to actual objects where they belong.

DefineIAction(Aload2)
  execAction() {
    abool.value = nil; // should be nil
    avarlist.value = ['bogus data 1','bogus data 2']; // should be []
    aintlist.value = []; // should be [3,3,4,4,5,5]
    astrlist.value = []; // should be ['asdf','qwer']
    avar.value = 0; // should be 3
    aint.value = 0; // should be 4
    astr.value = ''; // should be 'asdf'
    aboollist.value = []; // should be [nil]
    
    "BEFORE:";
    "<p>";
    "abool.value = <<toString(abool.value)>>\n";
    "avarlist.value = <<sayList(avarlist.value)>>\n";
    "aintlist.value = <<sayList(aintlist.value)>>\n";
    "astrlist.value = <<sayList(astrlist.value)>>\n";
    "avar.value = <<toString(avar.value)>>\n";
    "aint.value = <<toString(aint.value)>>\n";
    "astr.value = <<toString(astr.value)>>\n";
    "aboollist.value = <<sayList(aboollist.value)>>\n";
    
    //achievementScopeFileManager.loadObjectDataFromFile();
    achievementScopeManager.loadData();
    
    "<p>";
    "AFTER:";
    "<p>";
    "abool.value = <<toString(abool.value)>>\n";
    "avarlist.value = <<sayList(avarlist.value)>>\n";
    "aintlist.value = <<sayList(aintlist.value)>>\n";
    "astrlist.value = <<sayList(astrlist.value)>>\n";
    "avar.value = <<toString(avar.value)>>\n";
    "aint.value = <<toString(aint.value)>>\n";
    "astr.value = <<toString(astr.value)>>\n";
    "aboollist.value = <<sayList(aboollist.value)>>\n";
  }
  sayList(lst) {
    local str = '[';
    str += toString(lst);
//    for (local i = 1; i <= lst.length; i++) {
//      if (i > 1) str += ',';
//      str += toString(lst[i]);
//    }
    str += ']';
    return str;
  }
  actionTime = 0
;

VerbRule(Aload2)
  'aload2'
  : Aload2Action
  verbPhrase = 'aload/aloading Step 2'
;


// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: ACHIEVE (magically achieve an achievement)
// -----------------------------------------------------------------------------

VerbRule(Achieve)
  'achieve' singleLiteral
  : AchieveAction
  verbPhrase = 'achieve/achieving (what)'
;

DefineLiteralAction(Achieve)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    
    try {
      //local ach = Compiler.eval(objectName);
      local ach = achievementScopeUniquenessGuarantor[objectName];
      if (ach == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      "Starting value: ";
      "\n<<objectName>>.isAchieved = <<ach.isAchieved ? 'true' : 'nil'>>. ";
      ach.achieve();
      "<p>Okay, the achievement <<objectName>>
        has been achieved. ";
      "<p><<objectName>>.isAchieved = <<ach.isAchieved ? 'true' : 'nil'>>. ";
    }
    catch (Exception ex) {
      "Error in trying to achieve the achievement <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "ACHIEVE: Magically achieve an achievement. 
      <p>Example usage:
        \nACHIEVE achievementName (case-sensitive) ";
  }
  actionTime = 0
;

// To give a better response to just 'ACHIEVE'

DefineIAction(AchieveIntransitive)
  execAction() {
    AchieveAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(AchieveIntransitive)
  'achieve'
  : AchieveIntransitiveAction
  verbPhrase = 'achieve/achieving'
;



// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: UNACHIEVE (magically unachieve an achievement)
// -----------------------------------------------------------------------------

VerbRule(Unachieve)
  'unachieve' singleLiteral
  : UnachieveAction
  verbPhrase = 'unachieve/unachieving (what)'
;

DefineLiteralAction(Unachieve)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    
    try {
      //local ach = Compiler.eval(objectName);
      local ach = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (ach == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      "Starting value: ";
      "\n<<objectName>>.isAchieved = <<ach.isAchieved ? 'true' : 'nil'>>. ";
      ach.setAchieved(nil); // unachieve the achievement
      ach.set(&achievementNotified, nil);
      "<p>Okay, the achievement <<objectName>>
        has been unachieved. ";
      "<p><<objectName>>.isAchieved = <<ach.isAchieved ? 'true' : 'nil'>>. ";
      achUnachiever.achieve();
    }
    catch (Exception ex) {
      "Error in trying to unachieve the achievement <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "UNACHIEVE: Magically unachieve an achievement. 
      <p>Example usage:
        \nUNACHIEVE achievementName (case-sensitive) ";
  }
  actionTime = 0
;

// To give a better response to just 'UNACHIEVE'

DefineIAction(UnachieveIntransitive)
  execAction() {
    UnachieveAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(UnachieveIntransitive)
  'unachieve'
  : UnachieveIntransitiveAction
  verbPhrase = 'unachieve/unachieving'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: ERASE ALL (magically delete all achievement-scope data!)
// -----------------------------------------------------------------------------

DefineIAction(EraseAll)
  execAction() {
    try {
      local fn = new FileName(achievementScopeFileManager.achievementFilename);
      try {
        fn.deleteFile();
        "Okay, you asked for it. The achievement-scope file 
          <<achievementScopeFileManager.achievementFilename>> 
          has been deleted! ";
      }
      catch (Exception ex) {
        "There was a problem in trying to delete the achievement-scope file.
          <p>Exception message: <<ex.errmsg_>>\n";
      }
    }
    catch (Exception ex) {
      "Error in trying to delete 
        <<achievementScopeFileManager.achievementFilename>>. ";
      "<p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  actionTime = 0
;

VerbRule(EraseAll)
  'erase' 'all'
  : EraseAllAction
  verbPhrase = 'erase/erasing all'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: LOSE (magically lose the game)
// -----------------------------------------------------------------------------

DefineIAction(Lose)
  execAction() {
    "Okay, you lose. ";
    
    // endGame is a function we've defined above. See it for how it deals
    // with the "Loser"/"Winner" achievements.
    endGame(ftFailure);
  }
  actionTime = 0
;

VerbRule(Lose)
  'lose'
  : LoseAction
  verbPhrase = 'lose/losing'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: WIN (magically win the game)
// -----------------------------------------------------------------------------

DefineIAction(Win)
  execAction() {
    "Okay, you win. ";
    
    // endGame is a function we've defined above. See it for how it deals
    // with the "Loser"/"Winner" achievements.
    endGame(ftVictory);
  }
  actionTime = 0
;

VerbRule(Win)
  'win'
  : WinAction
  verbPhrase = 'win/winning'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: ASDF
// -----------------------------------------------------------------------------

DefineIAction(Asdf)
  execAction() {
    "Asdf!!! ";
    achAsdf.reveal();
    if (!asdfInfo.asdfHintGiven) {
      "<p>[There are some who say that good things happen to those who say ASDF 
        at least five times. Then again, there are some who say Limburger cheese
        is all right.] ";
      asdfInfo.asdfHintGiven = true;
    }
    asdfInfo.asdfCount++;
    if (asdfInfo.asdfCount >= 5) {
      achAsdf.achieve();
    }
  }
  actionTime = 0
;

VerbRule(Asdf)
  'asdf'
  : AsdfAction
  verbPhrase = 'asdf/asdfing'
;

asdfInfo: object
  // Okay to store these values to any conventional object, as opposed to a 
  // transient object or an achievement-scope variable, as we're trying to keep 
  // track of ASDFs typed *in a given playthrough*. 
  asdfCount = 0
  asdfHintGiven = nil
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: REVEAL (magically reveal an achievement)
// -----------------------------------------------------------------------------

VerbRule(Reveal)
  'reveal' singleLiteral
  : RevealAction
  verbPhrase = 'reveal/revealing (what)'
;

DefineLiteralAction(Reveal)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    
    try {
      //local ach = Compiler.eval(objectName);
      local ach = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (ach == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      "Starting value: ";
      "\n<<objectName>>.isVisible = <<ach.isVisible ? 'true' : 'nil'>>. ";
      ach.setVisible(); // make it visible
      "<p>Okay, the achievement <<objectName>>
        has been revealed. ";
      "<p><<objectName>>.isVisible = <<ach.isVisible ? 'true' : 'nil'>>. ";
    }
    catch (Exception ex) {
      "Error in trying to reveal the achievement <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "REVEAL: Magically reveal an achievement that may be initially secret. 
      <p>Example usage:
        \nREVEAL achievementName (case-sensitive) ";
  }
  actionTime = 0
;

// To give a better response to just 'REVEAL'

DefineIAction(RevealIntransitive)
  execAction() {
    RevealAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(RevealIntransitive)
  'reveal'
  : RevealIntransitiveAction
  verbPhrase = 'reveal/revealing'
;

// -----------------------------------------------------------------------------
// Verbs/actions: Achievements: CONCEAL (magically conceal an achievement)
// -----------------------------------------------------------------------------

VerbRule(Conceal)
  'conceal' singleLiteral
  : ConcealAction
  verbPhrase = 'conceal/concealing (what)'
;

DefineLiteralAction(Conceal)
  execAction() {
    local str = getLiteral();
    local tokens = str.split(' ');
    if (tokens.length != 1) {
      sayExampleUsage;
      return;
    }
    
    local objectName = tokens[1];
    
    try {
      //local ach = Compiler.eval(objectName);
      local ach = achievementScopeUniquenessGuarantor.asObjectsPerName[objectName];
      if (ach == nil) {
        throw new Exception('No unique object found for object name: ' + objectName);
      }
      
      "Starting value: ";
      "\n<<objectName>>.isVisible = <<ach.isVisible ? 'true' : 'nil'>>. ";
      ach.setVisible(nil); // make it visible
      "<p>Okay, the achievement <<objectName>>
        has been revealed. ";
      "<p><<objectName>>.isVisible = <<ach.isVisible ? 'true' : 'nil'>>. ";
    }
    catch (Exception ex) {
      "Error in trying to reveal the achievement <<objectName>>. 
      <p>Exception message: <<ex.errmsg_>>\n";
    }
  }
  sayExampleUsage {
    "CONCEAL: Magically conceal an achievement that may initially not be secret. 
      <p>Example usage:
        \nCONCEAL achievementName (case-sensitive) ";
  }
  actionTime = 0
;

// To give a better response to just 'CONCEAL'

DefineIAction(ConcealIntransitive)
  execAction() {
    ConcealIntransitiveAction.sayExampleUsage();
  }
  actionTime = 0
;

VerbRule(ConcealIntransitive)
  'conceal'
  : ConcealIntransitiveAction
  verbPhrase = 'conceal/concealing'
;

