#charset "us-ascii"

/* 
 *  Copyright (c) 2004 - 2005 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 London Cabs Library Extension
 *
 *  cabs.t
 *
 *  Defines the london cab class and growler and hansom cab
 *  instances. 
 */

#include <adv3.h>
#include <en_us.h>

class LondonCab: Vehicle, Chair
{
    bulkCapacity = 2000

    cabTravelMsg    = ''
    cabNoTravelMsg  = ''
    cabDepartMsg    = &cabDepartMsg
    cabDrivesOffMsg = &cabDrivesOffMsg

    actorInPrep     = (cabLanguageGlobals.actorInPrep)

    /*
     *  Method for calling the cab to the given destination. If
     *  the player isn't in the destination then no cab arrives
     *  and we return nil; otherwise we move the cab to the 
     *  destination and return true;
     */
    calledToLoc(dest)
    {
        /*
         *  if the player isn't in the destination
         *  then we don't move the cab there.
         */
        if (gPlayerChar.isIn(dest))
        {
            callWithSenseContext(self, sight, new function() {
                moveIntoForTravel(dest);
                reportAfter(&announceCabArrival, self);
            });

            return true;
        }
        else
            return nil;
    }

    travelToLoc(dest)
    {
        callWithSenseContext(self, sight, new function() {
            moveIntoForTravel(dest);

            mainReport(cabTravelMsg);

            gPlayerChar.lookAround(true);
        });
    }

    leaveLoc()
    {
        callWithSenseContext(self, sight, new function() {
            mainReport(cabDepartMsg);
            moveIntoForTravel(nil);
        });
    }

    driveOff()
    {
        callWithSenseContext(self, sight, new function() {
            mainReport(cabDrivesOffMsg, self);
            moveIntoForTravel(nil);
        });
    }
}

/*
 *  A hansom cab arrives when the player blows the whistle
 *  twice consecutively. The cab is reliable.
 */
hansomCab: LondonCab
{
    vocabWords  = (cabLanguageGlobals.hansomCabVocab)
    name        = (cabLanguageGlobals.hansomCabName)

    desc        = "<<mainReport(&hansomDesc)>>"

    /*
     *  The hansom won't go anywhere if the it's already
     *  at the destination.
     */
    travelToLoc(dest)
    {
        if (location == dest)
            mainReport(cabNoTravelMsg);
        else
            inherited(dest);
    }

    cabTravelMsg    = &hansomCabTravel
    cabNoTravelMsg  = &hansomCabNoTravel
}

/*
 *  A growler cab arrives when the player doesn't blow
 *  the whistle twice consecutively. The cab is unreliable.
 */
growlerCab: LondonCab
{
    vocabWords  = (cabLanguageGlobals.growlerCabVocab)
    name        = (cabLanguageGlobals.growlerCabName)

    desc        = "<<mainReport(&growlerDesc)>>"

    /*
     *  The growler cab travels to a random destination
     *  that was not what the player requested.
     */
    travelToLoc(dest)
    {
        local locList = [], newLoc;
        
        for (local o = firstObj(DestTravelTopic); o != nil; 
            o = nextObj(o, DestTravelTopic))
            if (o.dest != dest)
                locList += o.dest;

        newLoc = rand(locList);

        inherited(newLoc);
    }

    cabTravelMsg = &growlerCabTravel
}

cabLantern: MultiInstance
{
    locationList = [growlerCab, hansomCab]
    instanceObject: Fixture, LightSource, Thing 
    { 
        vocabWords  = (cabLanguageGlobals.cabLanternVocab)
        name        = (cabLanguageGlobals.cabLanternName)
    }
}