#charset "us-ascii"

/* 
 *  Copyright (c) 2008 by Kevin Forchione. All rights reserved.
 *   
 *  This file is part of the TADS 3 ADV3 Rulebooks library extension
 *
 *  arl_mods.t
 */

#include <t3.h>
#include <tads.h>
#include <adv3.h>

modify Action
{
    /*
     *  Overrides this method to allow calls to the various rules tables
     *  at important hook junctures in the command execution cycle.
     */
    doActionOnce()
    {
        /*
         *   Perform the sequence of operations to execute the action.  If
         *   an ExitSignal is thrown during the sequence, skip to the
         *   end-of-turn processing.  
         */
        try
        {
            local result;
            local impReport;

            /*
             *  If we match an "instead of" action rule then 
             *  we exit the process.
             */
            if (!InsteadOfActionRulebook.eval())
                /*
                 *   Before doing any actual execution, check the command for
                 *   remapping.  If we end up doing any remapping, the
                 *   remapping routine will simply replace the current command,
                 *   so we the remapping call will terminate the current action
                 *   with 'exit' and thus never return here.  
                 */
                checkRemapping();
            else    
                exit;

            /*
             *   If this is an implicit action, check for danger: we never
             *   try a command implicitly when the command is obviously
             *   dangerous.
             */
            if (isImplicit)
            {
                /* 
                 *   verify the action for an implicit command, checking
                 *   for actions that aren't allowe implicitly - we never
                 *   try a command implicitly when the command is (or
                 *   should be) obviously dangerous or is simply
                 *   non-obvious 
                 */
                result = verifyAction();

                /* 
                 *   If the action can be performed, but can't be performed
                 *   implicitly, abort.  Note that we only silently ignore
                 *   the action if it is allowed normally but not
                 *   implicitly: if it's not even allowed normally, we'll
                 *   simply fail later with the normal failure message,
                 *   since there's no harm in trying.  
                 */
                if (result != nil
                    && result.allowAction
                    && !result.allowImplicit)
                    abortImplicit;
            }

            /*
             *   If this is an implicit command, display a message
             *   indicating that we're performing the command.
             */
            impReport = maybeAnnounceImplicit();

            /*
             *   Make one or two passes through verifications and
             *   preconditions.  If any precondition performs an implicit
             *   command, we must run everything a second time to ensure
             *   that the implicit command or commands did not invalidate
             *   any earlier precondition or a verification.
             *   
             *   Run verifications before preconditions, because there
             *   would be no point in applying implicit commands from
             *   preconditions if the command verifies as illogical in the
             *   first place.  
             */
            for (local iter = 1 ; iter <= 2 ; ++iter)
            {
                /* verify the action */
                result = verifyAction();
                
                /* 
                 *   if verification doesn't allow the command to proceed,
                 *   show the reason and end the command 
                 */
                if (result != nil && !result.allowAction)
                {
                    /* show the result message */
                    result.showMessage();
                    
                    /* mark the command as a failure */
                    gTranscript.noteFailure();

                    /* 
                     *   if we have an implicit report, mark it as a mere
                     *   attempt, since the action can't be completed 
                     */
                    if (impReport != nil)
                        impReport.noteJustTrying();
                    
                    /* terminate the command */
                    exit;
                }

                /* 
                 *   Check preconditions of the action.  If we don't invoke
                 *   any implicit commands, we can stop here: nothing in
                 *   the game state will have changed, so there is no need
                 *   to re-verify or re-check preconditions.
                 *   
                 *   Only allow implicit actions on the first pass.  Do not
                 *   perform implicit actions on the second pass, because
                 *   if we did so we could get into an infinite loop of
                 *   conflicting preconditions, where each pass would
                 *   reverse the state from the last pass.  
                 */
                if (!checkPreConditions(iter == 1))
                    break;
            }

            /* 
             *   Disable sense caching once we start the action phase -
             *   once we start making changes to game state, it's too much
             *   work to figure out when to invalidate the cache, so simply
             *   turn off caching entirely.
             *   
             *   Note that the sense cache will already be disabled if we
             *   executed any implied commands, because the first implied
             *   command will have disabled the cache as soon as it reached
             *   its execution phase, and no one will have turned caching
             *   back on.  It does no harm to disable it again here.  
             */
            libGlobal.disableSenseCache();

            /* Ask if the rules table wants to handle it */
            if (!BeforeActionRulebook.eval())
            {
                /* run the before-action processing */
                beforeAction();
                
                /* 
                 *   notify the actor's containers that an action is about to
                 *   take place within them 
                 */
                gActor.forEachContainer(callRoomBeforeAction);

                /* call beforeAction for each object in the notify list */
                notifyBeforeAfter(&beforeAction);
            }
                
            /* 
             *   Invoke the action's execution method.  Catch any "exit
             *   action" exceptions - these indicate that the action is
             *   finished but that the rest of the command processing is to
             *   proceed as normal.  
             */
            try
            {
                /* Ask if the rules table wants to handle it */
                if (!ActorActionRulebook.eval())
                    /* notify the actor of what we're about to do */
                    gActor.actorAction();

                /* Ask if the rules table wants to handle it */
                if (!CheckActionRulebook.eval())
                    /* check the action */
                    checkAction();

                /* Ask if the rules table wants to handle it */
                if (!InsteadOfExecActionRulebook.eval() 
                    && !ExecActionRulebook.eval())
                    /* execute the action */
                    execAction();
            }
            catch (ExitActionSignal eaSig)
            {
                /* 
                 *   an exit action signal was thrown - since we've now
                 *   skipped past any remaining action processing, simply
                 *   continue with the rest of the command processing as
                 *   normal 
                 */
            }
            
            /* Ask if the rules table wants to handle it */
            if (!AfterActionRulebook.eval())
            {
                /* call afterAction for each object in the notify list */
                notifyBeforeAfter(&afterAction);
                
                /* notify the actor's containers of the completed action */
                gActor.forEachContainer(callRoomAfterAction);
            
                /* run the after-action processing */
                afterAction();
            }
        }
        catch (ExitSignal exc)
        {
            /* the execution sequence is finished - simply stop here */
        }
    }
}