#include <adv3.h>
#include <en_us.h>

splitGameLib: ModuleID
  name = 'TADS3 SplitGame Library Module'
  byline = 'by Chris Odhner <trask@tinaa.com>'
  htmlByline = 'by Chris Odhner <a href="mailto:trask@tinaa.com">&lt;trask@tinaa.com&gt;</a>'
  version = '0.9'
;
splitGameIdea: ModuleID
  name = 'Split Game Concept'
  byline = 'by Jan Thorsby <no_jthorsby_spam@broadpark.no>'
  htmlByline = 'by Jan Thorsby <a href="mailto:no_jthorsby_spam@broadpark.no">&lt;no_jthorsby_spam@broadpark.no&gt;</a>'
  version = '1.0'
  showAbout()
  {
    "<.p>In a Usenet news posting on rec.arts.int-fiction, Jan Thorsby wrote:
    <blockquote><em>6. This is a really stupid idea. The player plays two
    games at once. For instance in one game he plays a barbarian warrior
    fighting monsters, and in the other he plays a young man who is meeting
    the family of his girlfriend for the time and is trying to make a good
    impression. So if a mad dog attacks the barbarian, the player might type
    <Q>kill dog</Q> and the barbarian will kill the dog. But the young man will
    also kill the family poodle and so it is game over.</em></blockquote><.p>
    People immediately loved the idea, and several folks started trying to
    code it up.<.p>";
  }
;

/********************************************************
 *                                                      *
 * How to use this Module, the short version:           *
 *  1) Create two ActorInstance objects named me1 and   *
 *     me2, and a ActorInstanceDriver named god.        *
 *  2) Set initialPlayerChar to god.                    *
 *  3) Create two maps full of rooms, one for each of   *
 *     me1 and me2. Ensure these maps do not connect.   *
 *  4) Create a disconnected room and put god in it.    *
 *  5) me1 and me2 are your player characters, one for  *
 *     each map. They will each react to every command  *
 *     that the player types.                           *
 *  6) The game being played by me1 will be on the left *
 *     side of the screen, while me2 will be on the     *
 *     right side of the screen.                        *
 *  7) me1 lives in world1, me2 lives in world2. For    *
 *     each NPC you include in your game, you need to   *
 *     give them a whichWorld property (just a plain    *
 *     number) of either 1 or 2 so that they know who   *
 *     the playerChar is in their world.                *
 *                                                      *
 *  All over this file I call me1 and me2 different     *
 *  things. I call them robots, zombies, puppets,       *
 *  etc. So when you are reading the comments, just     *
 *  know that those all mean the same thing -- robot    *
 *  and zombie and puppet all mean ActorInstance.       *
 *                                                      *
 *  Keep in mind that every time you output text, you   *
 *  need to select which window it goes to. If you do   *
 *  nothing special, it will go to the correct window   *
 *  as long as it's happening in response to a command  *
 *  that a player types. But if you have any kind of    *
 *  daemons or scheduled events or anything, you need   *
 *  to handle it yourself... I was going to take it on  *
 *  and do "replace eventManager" but it turned out     *
 *  that I'm simply way too lazy.                       *
 *                                                      *
 *  I leave it to individual game authors to modify     *
 *  the showStatusLeft, showStatusRight, or             *
 *  showStatusHTML methods of the statusLine object to  *
 *  get whatever effect you desire. :-)                 *
 *                                                      *
 ********************************************************/

 /*
  * Version History
  *
  * 0.0 - 12-15 hours of work before I thought I'd covered everything.
  * 0.1 - Fixed errors (2) and warnings (14) from compiler; still useless.
  * 0.2 - Got display code working, made sample game, first real test run.
  *       Encountered an extremely amusing bug due to the fact that I was
  *       moving gPlayerChar around correctly, but forgot to ever move
  *       gActor to me1 and me2 when they attempted to do things.
  * 0.3 - After about four hours with the debugger, located and killed
  *       a bug that was causing an infinite loop when one (but not both)
  *       of the player characters attempted to act on an item that was
  *       out of scope. We can still get into this loop, but now if the
  *       loop keeps going for more than a thousand iterations, we break
  *       out of it. I tested and 1000 times through the loop doesn't 
  *       take much time, so I don't recommend reducing the loop count.
  * 0.4 - Replaced withCommandTranscript() so that it always uses a 
  *       fresh transcript, even if gTranscript is non-nil and active.
  *       This way, me1 and me2 get their own private transcripts (each
  *       writing to the correct bannerwindow) so we can avoid the 
  *       problem where you type 'get sword', see no output at all in
  *       the left or right bannerwindows, and get "Taken. Taken. " in
  *       god's tiny little commandline window, which doesn't have room
  *       for it. 
  * 0.5 - Fixed a bug that was causing undo to only half-work.
  * 0.6 - More cosmetic improvements, fixed "Nothing Obvious Happens".
  * 0.7 - Sorry. We can't allow 'oops' to function normally. I don't
  *       have time to fix it so that it works properly; I'm just going
  *       to disable it entirely.
  * 0.8 - Likewise we need to disable interactive disambiguation; praise
  *       MJR for providing good default responses for when disambiguation
  *       fails!
  * 0.9 - Cleaned up quit and restore processing. Seems pretty stable now.
  *
  */


/* WARNING: Things that might cause serious trouble with this
 * code...
 *
 * SpecialTopic entries in a NPC convNode. Avoid them if at all
 * possible... just stick with standard ask/tell/show/give/yes/no
 * and everything should be fine. If you absolutely MUST use them,
 * be aware that they might only work for the actor that is the
 * initialPlayerChar defined in gameMain; the other actor will
 * likely be unable to invoke the SpecialTopic regardless of 
 * what syntax they use, and the behaviour of the second actor
 * when the first actor triggers a SpecialTopic is difficult
 * to predict. Leave it alone, I say. :-)
 *
 * Again should work just fine.
 *
 * Quit will execute in the context of one of the actors,
 * and if it completes we don't care about the other one.
 * But if they decline at the are-you-sure prompt, then
 * the other actor is going to execute their own quit and
 * you need to decline again. Since most people don't 
 * type 'quit' unless they really want to, I don't think
 * this is much of an issue. We might want to check and
 * make sure that 'q' is NOT defined as a quit synonym.
 *
 * Pause will pause twice.
 *
 * Verbose and Terse are global; there will be no
 * independent control for each sub-game.
 *
 * Score will display the score twice, once in each
 * window. But if you check for the identity of 
 * gPlayerChar while you are printing out the score,
 * you could take advantage of this. 
 *
 * Notify - like verbose/terse... no independent 
 * control per subgame. One thing I haven't looked 
 * into yet is if the notifications will show up in
 * both windows, or only the appropriate one. I think
 * it will magically work the way you would want it 
 * to, with the score notifications going to the 
 * window of the subgame that generated the score
 * change. But I haven't tested this.
 *
 * Version, Credits, About - Will display in both
 * windows, like score. Again, you could detect the
 * active character and display appropriate information
 * for each subgame.
 *
 * Script: Painful. I'll try to redirect this one to the
 * InstanceDriver and handle duplicates. I have no idea
 * what the output would look like... it might even be
 * empty, because I don't know if script captures things
 * that happen in banner windows. I mean, it doesn't 
 * capture the stuff that happens on the status line,
 * right? This needs investigation.
 *
 * Record: Should be ok if I can redirect to god.
 * Of course, it might be difficult to redirect it
 * to god. Ugh.
 *
 * Replay: See record. Might need special handling for
 * if we just started the replay and try to start it
 * again. I mean, what happens to the current command
 * queue when you start a replay? Does it clear? I 
 * need to look at this.
 * 
 * Footnote: It would take quite a bit of hacking to get this
 * to work the way I would want, which would be shared control
 * over footnote verbosity, but completely separate footnote
 * lists for each story. As it is, if you put footnotes in both, 
 * you will get gaps in the footnote numbering; that is you might
 * see [1], [2], [4], [7], and [8] within the left-side story, 
 * while seeing [3], [5], [6], and [9] within the right-side
 * story. Even worse than that is that when you type 'note 2',
 * it will show note 2 in both windows, which means that you
 * are passing story information to the wrong window and 
 * totally destroying the effect of the game. The only way
 * I could see to use them at all without some extensive fixes
 * would be to write the footnote text so that it makes perfect
 * sense in both worlds... but in order to do that they would
 * probably need to be so bland that it wouldn't be worth the
 * trouble.
 *
 * If you are actually using this code to write a full-length
 * game, and you really want functional footnotes, let me
 * know and I will do the work.
 *
 * Save - I'll set up a PreSaveObject to clear out all three
 * command queues before saving. That should pretty much 
 * take care of it. Worst case someone has to re-type a
 * command after saving, if they did something silly like
 * "save, then go north"
 * 
 * Restore - I'll have to make sure that after a restore, 
 * our banner windows are set up correctly and that our
 * command queues are all empty. If the restore operation
 * is cancelled, we might get prompted a second time for
 * a save file to restore from. That's a bug, but I'm
 * not going to worry about it for the moment.
 *
 * Undo - we make a savepoint and bob goes north. we
 * make a savepoint and alice goes north. On the next
 * turn, we type undo. Bob's undo executes and restores
 * the savepoint prior to alice's movement; this nukes
 * the command queue so alice's undo never triggers.
 * We are left in a state where The undo command only
 * undid half of a turn. The solution to this, I think,
 * is to rig something up so that only one of the robots
 * (the one that goes first each turn) gets to create
 * savepoints. This requires a complete replacement 
 * of executeAction (from exec.t), despite the fact
 * that it's only a one-line change. Oops. Ok, tried that
 * and it didn't work. Indeed, creating the savepoint
 * only prior to me1's move allows us to go back far
 * enough in time to undo the moves of both me1 and me2.
 * The problem is that when we get back to that savepoint,
 * me2 still has commands in her queue. What we want to
 * do is trigger something to clear out her queue every
 * time we undo, after the undo is done with. 
 *
 * Restart - No problem, the first robot will do the
 * restart, and the second one will never know what
 * hit them. :-) Again, if the player answers "no"
 * to the are-you-sure question, they will get the
 * question again in the other window.
 *
 * Oops
 *
 */

splitGamePreInitStuff: object
  // this is where we handle the weird stuff.
  myPreSaveObj: PreSaveObject
  {
    execute()
    {
      // before saving the game, we want to clear
      // the command queues for both me1 and me2.
      me1.pendingCommand.setLength(0);
      me2.pendingCommand.setLength(0);
    }
  }
  myPostRestoreObj: PostRestoreObject
  {
    execute()
    {
     rightWindow.captureOutput({: libMessages.restoreOkay()}); 
     rightWindow.captureOutput({: say('\b')});
     rightWindow.captureOutput({: me2.lookAround(true)});
    }
  }
  me2FutureSelf: PostUndoObject
  {
    execute()
    {
      // Here's the sequence of events...
      // t0 - player types a comand.
      // t1 - god places the command in me1 and me2's queues.
      // t3 - me1 removes the command from his queue and creates savepoint
      // t4 - me1 executes command
      // t5 - me2 removes the command from her queue and executes it.
      // t6 - player types 'undo'
      // t8 - god places undo command in me1 and me2 queues.
      // t9 - me1 executes undo, taking us back to state of
      //      t3. Since the command is already removed from his
      //      queue, he does nothing during t4.
      // t10 - me2 removes the command (you know, the one we decided
      //       to UNDO) from her queue and executes it. Doh!
      //
      // So, the purpose of this PostUndoObject is to remove the command
      // from me2's command queue before she has a chance to reach step
      // t5. It's like a copy of me2 traveled back in time, to tell her
      // younger self "you don't want to do that, trust me". And you 
      // should always trust advice from your future self.
      me2.pendingCommand.setLength(0);
      //
      // See also the changes to executeAction below; they prevent me2
      // from creating savepoints during step t5, because if she did
      // make savepoints, me1's savepoints would always get overwritten
      // before he had a chance to type 'undo'.
      //
      // Ok, now that we undid, we want to print the same message on
      // me2's screen that already got printed for me1.
      rightWindow.captureOutput({:
      libMessages.undoOkay(libGlobal.lastActorForUndo,
      libGlobal.lastCommandForUndo)});
      rightWindow.captureOutput({: libGlobal.playerChar.lookAround(true)});
    }
  }
;

/*
 * OOPS isn't a normal command; if used as a verb under normal (non-oops)
 * circumstances it just prints a message saying you can't use oops now,
 * and the only time it is valid is during command line processing when
 * the parser finds a word that's not in it's vocabulary. The parser 
 * presents the user with a fake prompt, looks for an OOPS command, and
 * continues processing the current command if it finds one. If it finds
 * a non-OOPS command, it re-starts processing using the new command instead
 * of the old one. This presents us with several problems:
 *
 * The fake prompt is displayed to me1 before me2's turn starts executing;
 * this breaks the illusion of simultaneous moves.
 *
 * The fake prompt displays inside me1's window, but the input goes to god's
 * window. This makes things look really, really confusing and ugly.
 *
 * If me1 types OOPS SWORD and things continue, next me2 will get an OOPS
 * fake-prompt and could type OOPS POTATO. This provides an easy way to
 * "cheat" and issue commands to each actor independently, which defeats
 * the purpose of this entire module.
 *
 * Since it would be too much work to make oops work correctly, I'm taking
 * the easy way out and disabling it.
 */

replace tryOopsMain(tokList, issuingActor, targetActor,
            firstTokenIndex, cmdTokenList, cmdType)
{
    local unknownIdx;
    local w;

    /*
     *   Look for a word not in the dictionary. 
     */
    for (unknownIdx = nil, local i = 1, local len = tokList.length() ;
         i <= len ; ++i)
    {
        local cur;
        local typ;
        
        /* get the token value for this word */
        cur = getTokVal(tokList[i]);
        typ = getTokType(tokList[i]);
        
        /* check to see if this word is defined in the dictionary */
        if (typ == tokWord && !cmdDict.isWordDefined(cur))
        {
            /* note that we found an unknown word */
            unknownIdx = i;
            
            /* no need to look any further */
            break;
        }
    }

    /* 
     *   if we didn't find an unknown word, there's no need to offer the
     *   user a chance to correct a typo - simply return without any
     *   further processing 
     */
    if (unknownIdx == nil)
        return;

    /* get the unknown word, in presentable form */
    w = getTokOrig(tokList[unknownIdx]).htmlify();

    /* we can't do this interactively - treat it as a failure */
    throw new ParseFailureException(&wordIsUnknown, w);
}
modify libMessages
  commandResultsEmpty = ''
  oopsOutOfContext = "<.parser>This game, quite unfortunately, does not
  support the OOPS command. Sorry about that.<./parser> "
  confirmQuit()
  {
    // only me1 will ever reach this point, so we mirror the output
    // on me2's side of the world.
    quitMessageHack();
    rightWindow.captureOutput({: quitMessageHack()});
  }
  notTerminating()
  {
    // same as confirmQuit above, but now we have to strip
    // the QUIT command from me2's command list, or the
    // player will be asked are-you-sure again.
    notTerminatingHack();
    rightWindow.captureOutput({: notTerminatingHack()});
    me2.pendingCommand.removeElementAt(1);
  }
;
modify playerMessages
  wordIsUnknown(actor, txt)
  {
    "<.parser>The word <q><<txt>></q> is not necessary in this
        story.<./parser> ";
  }
;

quitMessageHack()
{
  "Do you really want to quit?\ (<<aHref('y', 'Y', 'Confirm quitting')>>
   is affirmative) >\ ";
}
notTerminatingHack()
{
  "<.parser>Continuing the story.<./parser>";
}

/*
 * We need to make a slight adjustment to Actor for correct NPC behavior. Many
 * times, an NPC will figure out how to react to another actor partly by
 * checking to see if the other actor is the player character. Also, many NPC
 * actions generate text output, and we need to know where to print that text.
 * Because of this, we  need to make sure that any time an NPC looks at me1 or
 * me2, they see a playerChar rather than something that looks like an NPC.
 * That's why we have a whichWorld property on all the NPCs.
 */
modify Actor
  whichWorld = nil // override this for all NPCs.
  executeTurn()
  {
    // save the current location of our immortal soul.
    local previousSoulLocation = gPlayerChar;
    // check what world we live in, and move the soul of
    // the player character into the correct body.
    // Then set the window context and fire off a turn.
    // Afterwards, restore the soul to god.
    local turnResult;
    switch(whichWorld)
    {
      case 1:
       gPlayerChar = me1;
       turnResult = leftWindow.captureOutput({: withActionEnv(EventAction, self,
         {: callWithSenseContext(isPlayerChar() ? nil : self, sight,
         {: executeActorTurn() }) })});
       gPlayerChar = previousSoulLocation;
       return turnResult;
      case 2:
      gPlayerChar = me2;
      turnResult = rightWindow.captureOutput({: withActionEnv(EventAction, self,
         {: callWithSenseContext(isPlayerChar() ? nil : self, sight,
         {: executeActorTurn() }) })});
       gPlayerChar = previousSoulLocation;
       return turnResult;
      case nil:
        // someone forgot to set the whichWorld property, or alternately
        // the current actor is god. It's certainly not me1 or
        // me2 because they override this method. We just do
        // nothing and hope for the best.
       return withActionEnv(EventAction, self,
          {: callWithSenseContext(isPlayerChar() ? nil : self, sight,
          {: executeActorTurn() }) });
      default:
        // we should never get here... throw an error message.
        "ERROR: <<name>> while executing turn thinks world is <<whichWorld>>\n";
        // but continue anyway, things probably aren't too broken.
        return nil;
    }
  }
;
  

/****************************************************************
 * THIS IS EXACTLY LIKE THE LIBRARY VERSION EXCEPT FOR ONE LINE *
 ****************************************************************/
replace executeAction(targetActor, targetActorPhrase,
  issuingActor, countsAsIssuerTurn, action)
  {
    local results;
startOver:
    results = new BasicResolveResults();
    results.setActors(targetActor, issuingActor);
    try
    {
        action.resolveNouns(issuingActor, targetActor, results);
    }
    catch (RemapActionSignal sig)
    {
        sig.action_.setRemapped(action);
        action = sig.action_;
        goto startOver;
    }
    if (action.includeInUndo
        && action.parentAction == nil
        && (targetActor.isPlayerChar()
            || (issuingActor.isPlayerChar() && countsAsIssuerTurn)))
    {
        libGlobal.lastCommandForUndo = action.getOrigText();
        libGlobal.lastActorForUndo =
            (targetActorPhrase == nil
             ? nil
             : targetActorPhrase.getOrigText());
          if(gPlayerChar == me1)  /* <============= THERE IT IS */
            savepoint();
    }
    if (countsAsIssuerTurn && !action.isConversational(issuingActor))
    {
        issuingActor.lastInterlocutor = targetActor;
        issuingActor.addBusyTime(nil,
                                 issuingActor.orderingTime(targetActor));
        targetActor.nonIdleTurn();
    }
    if (issuingActor != targetActor
        && !action.isConversational(issuingActor)
        && !targetActor.obeyCommand(issuingActor, action))
    {
        if (issuingActor.orderingTime(targetActor) == 0)
            issuingActor.addBusyTime(nil, 1);
        throw new TerminateCommandException();
    }
    action.doAction(issuingActor, targetActor, targetActorPhrase,
                    countsAsIssuerTurn);
}



class ActorInstanceDriver: Actor

  /*
   * The ActorInstanceDriver is a strange and wonderful beast.
   * It doesn't actually do anything, it just splits off turns
   * so that two other Actors can both be the gPlayerChar at
   * the same time. In fact, any actors of this type should
   * be safely sequestered away in a well-lit room with no
   * exits and no objects possessing vocabulary. Most of the
   * things that normally happen during executeActorTurn may
   * be skipped here, because this Actor isn't supposed to
   * interact with the rest of the game at all. 
   */
  executeActorTurn()
  {   

    /* 
     *   If neither of our robots have pending commands
     *   to execute, our next task will be to read and execute a
     *   command 
     */
    if (me1.pendingCommand.length() == 0 && me2.pendingCommand.length() == 0)
    {
      local toks;
      local whatTheyTyped;
      
      /* Print a fake command prompt in each window. */
      leftWindow.captureOutput({: say('\n')});
      rightWindow.captureOutput({: say('\n')});
      leftWindow.captureOutput({: libMessages.mainCommandPrompt(rmcCommand)});
      rightWindow.captureOutput({: libMessages.mainCommandPrompt(rmcCommand)});
      /* read a command line and get the resulting token list */
      toks = readMainCommandTokens(rmcCommand);
      
      /* 
       *   re-activate the main transcript - reading the command
       *   line will have deactivated the transcript, but we want it
       *   active again now that we're about to start executing the
       *   command 
       */
      gTranscript.activate();
      
      /* 
       *   If it came back nil, it means that the input was fully
       *   processed in pre-parsing; this means that we don't have
       *   any more work to do on this turn, so we can simply end our
       *   turn now.  
       *
       *   Note -- in the default library, I couldn't find anything
       *   that would produce this nil toks condition. And I'm not
       *   adding anything that would do this. So it should never
       *   happen. Hopefully if it does ever happen, it's ok that
       *   none of the robots will get a turn.
       */
      if (toks == nil)
          return true;
      
      /* retrieve the token list from the command line */
      whatTheyTyped = toks[1];
      toks = toks[2];
      
      /* 
       *   Add it to our pending command queue for both robots.
       *   Since we want them to think that they are real playerChars,
       *   each with a human behind the keyboard, we set the issuer
       *   in each case to be the robot we are giving the command to.
       *   Little do they know that they are mere puppets.. Muahahahaha!!!
       *   
       *   Since this is a newly-read command line, we're starting a
       *   new sentence.  
       */
      me1.addPendingCommand(true, me1, toks);
      me2.addPendingCommand(true, me2, toks);

      /*
       * Before we execute the commands, let's fake a command prompt
       * in each of the windows. We want them to look and feel as much
       * like an independent game as possible. We already printed the
       * standard ">" prompt up above, so now we just add the text
       * of what they typed in the god window.
       */
       leftWindow.captureOutput({: fakeprompt(whatTheyTyped)});
       rightWindow.captureOutput({: fakeprompt(whatTheyTyped)});
       

      /*
       *   Ok, so here is the fun part. We call each robot and tell
       *   them to execute a turn. After they are done, we get back
       *   here and continue with some really important stuff.
       */
      leftWindow.captureOutput({: me1.executeInstanceTurn()});
      rightWindow.captureOutput({: me2.executeInstanceTurn()});
  
      /*
       *   Check to see if we have any pending command to execute.  If
       *   so, our next task is to execute the pending command.  
       *
       *   You may be asking yourself why we bother with this step,
       *   since we pass off all the commands to our zombies and
       *   we never add anything to our own queue. Good question...
       *   the problem comes in when we need to deal with system
       *   commands like undo, save, and so on. We can't detect
       *   them ourselves too easily, because we pass the commands
       *   out to the robots as token lists, before parsing occurs.
       *   We could do the parsing ourselves and then send the commands,
       *   which would be much simpler, but the problem with that is
       *   that then nothing would be in scope for any command because
       *   we are just an invisible, formless being floating in the
       *   void. So that would suck.
       *
       *   There's a lot of discussion on this subject above, and
       *   in the end even most special verbs are ok to be handled
       *   by one of the puppets. But there are a few commands that
       *   really need to be handled by god. In those cases, I think
       *   it will end up being fixed with modifications to the
       *   action objects so that they can be smart enough to do
       *   the playerChar switch before proceeding; but I can't
       *   say I'm positive that there will never be anything in
       *   this queue.
       */
      if (pendingCommand.length() != 0)
      {
          local cmd;
      
          /* remove the first pending command from our queue */
          cmd = pendingCommand[1];
          pendingCommand.removeElementAt(1);
      
          /* if this is a real command, note the non-idle turn */
          if (cmd.hasCommand)
              nonIdleTurn();
      
          /* execute the first pending command 
           * this ends up in exec.t with executeAction */
          cmd.executePending(self);
      
        /* 
         *   We're done with this turn.  If we no longer have any
         *   pending commands, tell the scheduler to refigure the
         *   execution order, since another object might now be ready
         *   to run ahead of our idle activity.  
         */
        if (pendingCommand.indexWhich({x: x.hasCommand}) == nil)
            return nil;
        else
            return true;
      }
      else
      {
        /* We skipped reading a command above because at least one
           of our ActorInstance guys already had commands in the
           queue. If we don't give them an opportunity to clear that
           up, we will never get our command prompt back. :-) */
        if (me1.pendingCommand != nil)   
          if (me1.pendingCommand.length() != 0)
            leftWindow.captureOutput({: me1.executeInstanceTurn()});

        if (me2.pendingCommand != nil)
          if (me2.pendingCommand.length() != 0)
            rightWindow.captureOutput({: me2.executeInstanceTurn()});

      }
  
      /* we have no pending work to perform, so take an idle turn */
      idleTurn();
  
      /* no change in scheduling priority */
      return true;
    }
    return true;
  }
  fakeprompt(whatTheyTyped)
  {
    // given an input string, this filters the input
    // to create a fake prompt string.
    "<font face=\"tads-input\"><<whatTheyTyped>></font>\n";
  }
;

class ActorInstance: Actor
    // Has god had a turn recently? We risk getting into an
    // infinite loop if, for example, me1 picks up an item
    // that doesn't exist in me2's world. me1 and god will
    // both think that they have had a turn, and so they
    // will increment nextRunTime. me2, however, won't have
    // consumed any time; the game assumes that "get rock"
    // when there aren't any rocks in the room shouldn't 
    // use up someone's turn. It just wouldn't be fair to
    // punish people for mis-typing, or assuming that an
    // item mentioned in a room description exists, when it
    // doesn't. But the problem is that if me1 and god have
    // already moved, but me2 hasn't consumed any time, the
    // scheduler keeps on coming back and giving me2 a turn,
    // and if me2 doesn't do anything during its turn, the
    // scheduler never advances the game clock so me1, and
    // more importantly god, never get to move. We have a
    // deadlock. So, to fix this, we want to make sure that
    // our nextRunTime gets incremented once in a while,
    // even if god didn't tell us to. We don't want to do
    // this every time executeTurn gets called, because 
    // that would create the unfair condition mentioned 
    // above and sometimes could cause NPCs to move many
    // times per (apparent) turn. Instead, we keep track
    // of how many times executeTurn has been called since
    // the most recent time we ran executeInstanceTurn;
    // if the number is large enough to indicate we are
    // in an infinite loop, we increment our nextRunTime.
    loopwatcher = 0
    executeTurn() 
    { 
      if(loopwatcher > 1000)
      {
        loopwatcher = 0;
        nextRunTime += 1;
      }
      else
      {
        loopwatcher += 1;
      }
    }
    // This replaces executeTurn()...
    executeInstanceTurn()
    {
      // God told us to run, so it's safe to reset loopwatcher.
      loopwatcher = 0;
      local tempresult;
      // This is basically just copied from the original
      // executeTurn, but we steal playerChar status first
      // and give it back to god afterward. God is responsible
      // for making sure the output of our turn goes to the
      // correct window.
      local originalPlayerCharacter = gPlayerChar;
      local originalActor = gActor;
      gPlayerChar = self;
      gActor = self;
      
      "<.commandsep>";
      // do our turn.
      tempresult = withActionEnv(EventAction, self,
       {: callWithSenseContext(isPlayerChar() ? nil : self, sight,
                {: executeActorTurn() }) });

       // Ok, give god his hat back.
       gPlayerChar = originalPlayerCharacter;
       gActor = originalActor;
       return tempresult;
    }

   executeActorTurn()
    {
    // This doesn't need much work at all for the split-screen stuff.
        /*
         *   If we have a pending response, and we're in a position to
         *   deliver it, our next work is to deliver the pending response.
         */
        if (pendingResponse != nil && canTalkTo(pendingResponse.issuer_))
        {
            /* 
             *   We have a pending response, and the command issuer from
             *   the pending response can hear us now, so we can finally
             *   deliver the response.
             *   
             *   If the issuer is the player character, send to the player
             *   using our deferred message generator; otherwise, call the
             *   issuer's notification routine, since it's an NPC-to-NPC
             *   notification.  
             */
            if (pendingResponse.issuer_.isPlayerChar())
            {
                /* 
                 *   we're notifying the player - use the deferred message
                 *   generator 
                 */
                getParserDeferredMessageObj().(pendingResponse.prop_)(
                    self, pendingResponse.args_...);
            }
            else
            {
                /* it's an NPC-to-NPC notification - notify the issuer */
                pendingResponse.issuer_.notifyIssuerParseFailure(
                    self, pendingResponse.prop_, pendingResponse.args_);
            }

            /* 
             *   in either case, we've gotten this out of our system now,
             *   so we can forget about the pending response 
             */
            pendingResponse = nil;
        }
            
        /* check to see if we're waiting for another actor */
        if (checkWaitingForActor())
        {
            /* 
             *   we're still waiting, so there's nothing for us to do; take
             *   an idle turn and return 
             */
            idleTurn();
            return true;
        }

        /*
         * If we just did an UNDO or RESTORE, it's possible to reach
         * this point with a nil pendingCommand ; if so just return
         * true and move on with life.
         */

         if (pendingCommand == nil)
           return true;
            
        /* 
         *   if we're the player character, and we have no pending commands
         *   to execute, our next task will be to read and execute a
         *   command ... NOT!!! we shouldn't ever read commands if the
         *   queue is empty; instead just call it an idle move.
         */
        if (pendingCommand.length() == 0 && isPlayerChar())
        {
                // same action main library takes if nil tokens.
                return true;
        }

        /*
         *   Check to see if we have any pending command to execute.  If
         *   so, our next task is to execute the pending command.  
         */
        if (pendingCommand.length() != 0)
        {
            local cmd;
            
            /* remove the first pending command from our queue */
            cmd = pendingCommand[1];
            pendingCommand.removeElementAt(1);
            
            /* if this is a real command, note the non-idle turn */
            if (cmd.hasCommand)
                nonIdleTurn();
            
            /* execute the first pending command */
            cmd.executePending(self);
            
            /* 
             *   We're done with this turn.  If we no longer have any
             *   pending commands, tell the scheduler to refigure the
             *   execution order, since another object might now be ready
             *   to run ahead of our idle activity.  
             */
            if (pendingCommand.indexWhich({x: x.hasCommand}) == nil)
                return nil;
            else
                return true;
        }
        
        /*
         *   If we're following an actor, and the actor isn't in sight, see
         *   if we can catch up.  
         */
        if (followingActor != nil
            && location != nil
            && (followingActor.location.effectiveFollowLocation
                != location.effectiveFollowLocation))
        {
            local info;
            
            /* see if we have enough information to follow */
            info = getFollowInfo(followingActor);
                
            /* 
             *   Check to see if we have enough information to follow the
             *   actor.  We can only follow if we saw the actor depart at
             *   some point, and we're in the same location where we last
             *   saw the actor depart.  (We have to be in the same
             *   location, because we follow by performing the same command
             *   we saw the actor perform when we last saw the actor
             *   depart.  Repeating the command will obviously be
             *   ineffective unless we're in the same location as the actor
             *   was.)  
             */
            if (info != nil)
            {
                local success;
                
                /* 
                 *   we know how to follow the actor, so simply perform
                 *   the same command we saw the actor perform.  
                 */
                newActorAction(self, Follow, followingActor);
                
                /* note whether or not we succeeded */
                success = (location.effectiveFollowLocation ==
                           followingActor.location.effectiveFollowLocation);
                    
                /* notify the state object of our attempt */
                curState.justFollowed(success);
                
                /* 
                 *   if we failed to track the actor, note that we are
                 *   unable to follow the actor 
                 */
                if (!success)
                {
                    /* note that we failed to follow the actor */
                    cannotFollow();
                }
                
                /* we're done with this turn */
                return true;
            }
            else
            {
                /* 
                 *   we don't know how to follow this actor - call our
                 *   cannot-follow handler 
                 */
                cannotFollow();
            }
        }

        /* we have no pending work to perform, so take an idle turn */
        idleTurn();
        
        /* no change in scheduling priority */
        return true;
    }
;

/*************************************************************
 *
 * Display mmanagement stuff... isn't it FUN!
 *
 *************************************************************/

// we can't let me1 and me2 piggyback on god's transcript,
// because output would go to the wrong windows. We also
// don't want to let multiple transcripts be active at a
// time. 
//
// Of course, we need to check if we already had a transcript;
// otherwise we risk a nil object reference when we try to
// enable or disable the old one.
replace withCommandTranscript(transcriptClass, func)
{
  local transcript;
  local oldTranscript;
  local oldTSActive;
  transcript = transcriptClass.createTransientInstance();
  oldTranscript = gTranscript;
  gTranscript = transcript;
  if(oldTranscript != nil)
  {
    oldTSActive = oldTranscript.isActive;
    oldTranscript.deactivate();
  }
  mainOutputStream.addOutputFilter(transcript);
  try
  {
    return (func)();
  }
  finally
  {
    mainOutputStream.removeOutputFilter(transcript);
    gTranscript = oldTranscript;
    transcript.showReports(true);
    if (oldTSActive)
      oldTranscript.activate();
  }
}

// This bannerwindow is for me1. It ends up being half as
// big as what's defined here because rightWindow steals
// half the space.
leftWindow: BannerWindow
  initBannerWindow()
  {
    // do nothing if this has already been done.
    if (inited_)
      return;

    // inherit initBannerWindow, which sets inited_ for us.
    inherited();

    // Here is where we set all the good stuff...
    // parent (nil for the system window)
    // where (first, last, before, after)
    // other (before or after *who*?)
    // type (text or textgrid)
    // align (Top, Bottom, Left, Right)
    // size (percent, absolute, contents-fit)
    // units (BannerSizePercent, BannerSizeAbsolute)
    // styleflags (BannerStyleXxx: Border, VScroll, HScroll,
    //             AutoVScroll, AutoHScroll, TabAlign, 
    //             MoreMode (implies AutoVScroll).
    showBanner(nil, BannerLast, nil, BannerTypeText, BannerAlignTop,
               75, BannerSizePercent, BannerStyleBorder   
               | BannerStyleAutoVScroll | BannerStyleVScroll);
  }
  active = true;
;
// Same setup as leftWindow, but align right and use leftwindow
// as our parent.
rightWindow: BannerWindow
  initBannerWindow()
  {
    if(inited_)
      return;
    inherited();
    showBanner(
      leftWindow, BannerLast, nil, BannerTypeText, BannerAlignRight,
      50, BannerSizePercent, BannerStyleBorder | BannerStyleVScroll
      | BannerStyleAutoVScroll);
  }
  active = true;
;

/***************************************************** 
 * Parser disambiguation stuff... we need to disable *
 * interactive disambiguation because it screws up   *
 * the user interface and breaks simultaneous action *
 * resolution; if one of the actors gets a prompt to *
 * disambiguate a command, and the other does not,   *
 * the actor with the disambiguation prompt can just *
 * type a completely different command, which ruins  *
 * the feeling of having them both always executing  *
 * the same commands.                                *
 *****************************************************/

modify BasicResolveResults
  canResolveInteractively(action) { return nil; }
;

