#charset "us-ascii"

/*

Tads-3 RAP Planbase version 1.3

RAP 1.0 Planbase, Copyright (c) by Nate Cull

TADS-3 port and revisions Copyright (c) 2002-2004 by Steve Breslin
(email: versim@hotmail.com)

Thanks goes to the v-space Tads-3 discussion list, particularly Eric
Eve and Mike Roberts, for advice regarding some of the plans.

License:

You can use this material however you want, but if you decide to make
public anything that uses, #include's or otherwise borrows from this
material, you have to make your improvements publicly available, and
advertise them to the IF community.

That way, you will help keep this up to date for everybody else, and
everybody else will help keep it up to date for you.

You may redistribute this verbatim or in modified form only if you
keep the copyrights and license intact.

Feel encouraged to release your source code along with your game,
though this isn't a requirement.



Default Planbase
----------------

Standard planbase definitions: commonly useful & general rSteps


defines:

  rFailedAction: rStep    // what to do if rFind doesn't find an action
  rHave: rStep            // hold item or make it visible in inventory
  rNotHave: rStep         // be not containing the object
  rHold: rStep            // hold the object directly in inventory
  rReachable: rStep       // make reachable the object
  rOpen: rStep            // open the object
  rUnlocked: rStep        // unlock the object
  rClosed: rStep          // close the object
  rLocked: rStep          // lock the object
  rObjIn: rStep2P         // put the object in another object
  rObjDirectlyIn: rStep2P // put the object directly in another object
  rObjOn: rStep2P         // put the object on another obj
  rObjDirectlyOn: rStep2P // put the object directly on another object
  rStandOn: rStep         // stand on an object (e.g., nestedroom)
  rTurnOn: rStep          // turn the object on
  rTurnOff: rStep         // turn the object off
  rPress: rStep           // press p (p is a button)  [not implemented]
  rIn: rStep              // be directly in a location
  rNotOwnedByOther: rStep // if possession of the object is blocked by
                          // another actor's possession, rAskFor it
  rAskFor: rStep          // ask another actor for the object, or its
                          // container, if object isn't visible.
  rGiveTo: rStep          // give requested item to another actor


  rMapHandler: object     // encapsulates RAP movement plan production
      rBuildMap: method   // creates default movement plans
      rBuildMapPlan: method // OVERRIDE THIS to add new default
                            // behavior for movement plans


modifies:

  room // adds rFromList and rMovePlan properties

*/

#include <adv3.h>
#include <en_us.h>


rFailedAction: rStep
    name = 'rNoAction'
    rTrue(a, p) { return nil; }
    rAction(a, p) {
        if (p[1].propDefined(&rAction, PropDefDirectly)) {

            /* Try to call the action method, so the game will report
             * why the command failed. This is mostly useful when the
             * RAP actor has been ordered to do something by the PC,
             * which the RAP processor can't figure out how to do.
             *
             * It is common to customize rIdleCode to switch plans when
             * the current plan isn't working; this works well with
             * fully automated NPC's. In this case, you'll want to
             * replace this rAction routine with a simple call to
             * a.rIdleCode().
             */
            p[1].rAction(a, p[2]);
        }
        else
            a.rIdleCode();
    }
;

// plan to make the parameter be in the actor's inventory.
rHave: rStep
    name = 'rHave'

    /* rTrue is true if the param is directly in the actor's inventory,
     * or if the param is visible within the containment of the actor's
     * inventory. These are two separate checks because the latter is
     * considerably less efficient, and considerably less frequent.
     */
    rTrue(a, p) {
        return ( (p.isDirectlyIn(a)) ||
                 (p.isIn(a) && !a.findVisualObstructor(p)) );
    }

    rPlans(a, p) {
        if (p.ofKind(Fixture))
            return [];
        return(
                [
                    [rBe, rNotOwnedByOther, p,
                     rBe, rReachable, p,
                     rDo, rHave, p]
                ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, Take, p);
    }
;

rNotHave: rStep
    name = 'rNotHave'

    rTrue(a, p) {
        return (!p.isIn(a));
    }

    rPlans(a, p) {
        return( [ [ rDo, rNotHave, p ] ] );
    }

    rAction(a, p) {
        nestedActorAction(a, Drop, p);
    }
;

// plan to make the parameter be directly in the actor.
// this is similar to rHave, but it requires that the actor be holding
// the item.
rHold: rStep
    name = 'rHold'

    /* rTrue is true only if the param is directly in the actor's
     * inventory.
     */
    rTrue(a, p) {
        if (!p.location)
            return nil;
        return (p.location == a);
    }

    rPlans(a, p) {
        if (p.ofKind(Fixture))
            return [];
        return(
                [
                  [rBe, rNotOwnedByOther, p,
                   rBe, rReachable, p,
                   rDo, rHold, p]
                ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, Take, p);
    }
;

// (rNotHold probably means rNotHave except in custom cases, so we don't
// define rNotHold separately in the standard planbase.)

rReachable: rStep
    name = 'rReachable'
    rTrue(a, p) {
        return (a.canTouch(p));
    }
    rPlans(a, p) {
        local ret, closedList = [];
        // if p has no location, no plan will work. Return no plan: []
        if (!p.location) return [];

        /* We do two things here. Firstly, we're looking for the
         * room object where p is. Second, we're producing a list of
         * closed objects which are between p and Room, on the
         * containment tree. These objects must be open before p is
         * reachable.
         */
        while (!p.ofKind(Room)) {
            p = p.location;
            if (p && p.ofKind(Openable) && !p.isOpen)
                closedList =+ [p];
        }

        /* If there are no containment barriers, being reachable simply
         * means being in the same room.
         */
        if (closedList == [])
            return ( [ [rBe, rIn, p] ] );

        /* We count backwards in the closedList we've created, adding a
         * step for opening each thing in turn. (We count backwards
         * because we built the list from inner to outer, but the
         * objects must be opened from outer to inner: open big box to
         * open small box, etc.)
         *
         * Note that we don't place a 'rBe rIn' entry before calling
         * 'rBe rOpen' because rOpen handles being in, and rOpen may
         * also need to handle getting a key before worrying about
         * making the target openable (and therefore reachable).
         */
        ret = [];
        for (local i = closedList.length() ; i > 0 ; i--)
            ret += [ [rBe, rOpen, closedList[i]] ];
        return ret;
    }
;

// handles both doors and openable containers.
rOpen: rStep
    name = 'rOpen'
    rTrue(a, p) {
        return (p.isOpen);
    }
    rPlans(a, p) {
        /* First we check if there's a custom plan defined by the
         * object.
         */
        if (p.propDefined(&rOpenCustomPlan)) {
            /* If the custom plan two takes arguments, we assume it
             * wants to know (a, p).
             */
            if (p.getPropParams(&rOpenCustomPlan)[1] == 2) {
                return p.rOpenCustomPlan(a, p);
            }
            /* We assume otherwise that the custom plan is a list, so
             * we just return its value.
             */
            return p.rOpenCustomPlan;
        }
        /* There's no custom plan defined by the object. Proceed with
         * our default plan: if it's locked, we first get the key.
         */
        if (p.isLocked)
            return (
                [ [rBe, rUnlocked, p,
                   rDo, rOpen, p] ] // (this step is unreachable.)
            );
        return (
            [ [rBe, rReachable, p,
               rDo, rOpen, p] ]
        );
    }
    rAction(a, p) {
        nestedActorAction(a, Open, p);
    }
;

rUnlocked: rStep
    name = 'rUnlocked'

    rTrue(a, p) {
        return (p.isLocked == nil);
    }

    rPlans(a, p) {
        local r = [];
        if (p.keyList) {
            // add a separate plan for each p.keyList entry
            for (local i = 1 ; i <= p.keyList.length() ; i++) {
                if (p.keyList[i].location)
                    r += [ [ rBe, rHave, p.keyList[i],
                             rBe, rReachable, p,
                             rDo, rUnlocked, [p, p.keyList[i]] ] ];
            }
            return r;
        }
        else
            /* Many keyless unlockables will want to customize this
             * plan. This is only a default best-guess.
             */
            return(
                    [ [ rBe, rReachable, p,
                        rDo, rUnlocked, p ] ]
            );
    }

    rAction(a, p) { // p = [object, key] || p = object
        if (p.ofKind(List))
            nestedActorAction(a, UnlockWith, p[1], p[2]);
        else
            nestedActorAction(a, Unlock, p);
    }
;

rClosed: rStep
    name = 'rClosed'

    rTrue(a, p) {
        return (!p.isOpen);            
    }

    rPlans(a, p) {
        return (
                [ [rBe, rReachable, p,
                   rDo, rClosed, p] ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, Close, p);
    }
;

rLocked: rStep
    name = 'rUnlocked'
    rTrue(a, p) {
        return (p.isLocked);
    }

    rPlans(a, p) {
        local ret = [];
        if (p.keyList) {
            for (local i = 1 ; i <= p.keyList.length() ; i++) {
                if (p.keyList[i].location) {
                    ret += [ [ rBe, rHave, p.keyList[i],
                             rBe, rReachable, p,
                             rDo, rLocked, [p, p.keyList[i]] ] ];
                }
            }
            return ret;
        }
        else
            return(
                [ [ rBe, rReachable, p,
                    rDo, rLocked, p] ]
            );
    }

    rAction(a, p) { // p = [object, key] || p = object
        if (p.ofKind(List))
            nestedActorAction(a, LockWith, p[1], p[2]);
        else
            nestedActorAction(a, Lock, p);
    }
;

/* concerns object containment, not the Actor's location. Use rIn
 * for automating actor movement. Use rObjIn, etc., for automating
 * actor's manipulation of objects.
 *
 * The planbase assumes 'p' is a list [A, B], where A is the object
 * to be put in B. It further assumes that A should remain where
 * it is. So if you want the Rapper to pick up the container object
 * on the way to the second object, you'll need to adjust the plan
 * accordingly: [rBe, rHave, p[2], rBe, rHave, p[1], rDo rObjIn, p]
 *
 * If B is in A, a problem arises: A cannot be put in B because B
 * is in A. The library doesn't feel comfortable assuming that the
 * actor should implicitly take B. So we have to incorporate this
 * contingency in the plan construction.
 *
 * You may then wish that the Actor drops B on the ground. We're
 * not assuming this, but it's easy enough for you to customize this
 * plan if that's your desired default behavior. With a little more
 * work, you could get the actor to replace object B in its original
 * location, but again, that is a bit too specialized for the standard
 * planbase.
 */

rObjIn: rStep2P
    name = 'rObjIn'

    rTrue(a, p) {
        return (p[1].isIn(p[2]));
    }

    rPlans(a, p) {
        local ret = [];
        if (p[2].isIn(p[1]))
            ret =+ [ rBe, rHold, p[2] ];
        return ( [ ret + [ rBe, rHave, p[1],
                           rBe, rReachable, p[2],
                           rDo, rObjIn, p ] ]
        );
    }

    rAction(a, p) { // p = [obj, location]
        nestedActorAction(a, PutIn, p[1], p[2]);
    }
;

rObjDirectlyIn: rStep2P
    name = 'rObjDirectlyIn'

    rTrue(a, p) {
        if (!p[1].location)
            return nil;
        return (p[1].location == p[2]);
    }

    rPlans(a, p) {
        return (rObjIn.rPlans(a, p));
    }
;

/* rObjOn is basically the same as rObjIn.
 * See the comments to rObjIn for relevant notes.
 */
rObjOn: rStep2P
    name = 'rObjOn'

    rTrue(a, p) {
        return (rObjIn.rTrue(a, p));
    }

    rPlans(a, p) {
        local ret = [];
        if (p[2].isIn(p[1]))
            ret =+ [ rBe, rHold, p[2] ];
        return ( [ ret + [ rBe, rHave, p[1],
                           rBe, rReachable, p[2],
                           rDo, rObjOn, p ] ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, PutOn, p[1], p[2]);
    }
;

rObjDirectlyOn: rStep2P
    name = 'rObjDirectlyOn'

    rTrue(a, p) {
        return (rObjDirectlyIn.rTrue(a, p));
    }

    rPlans(a, p) {
        return (rObjOn.rPlans(a, p));
    }
;

rStandOn: rStep
    name = 'rStandOn'

    rTrue(a, p) {
        if (!a.location)
            return nil;
        return (a.location == p);
    }

    rPlans(a, p) {
        return ( [ [rBe, rReachable, p,
                    rDo, rStandOn, p] ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, StandOn, p);
    }
; 

rTurnOn: rStep
    name = 'rTurnOn'

    rTrue(a, p) {
        return (p.isOn);       
    }

    rPlans(a, p) {
        return ( [ [rBe, rReachable, p,
                        rDo, rTurnOn, p] ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, TurnOn, p);
    }
;

rTurnOff: rStep
    name = 'rTurnOff'

    rTrue(a, p) {
        return (!p.isOn);
    }

    rPlans(a, p) {
        return ( [ [rBe, rReachable, p,
                        rDo, rTurnOff, p] ]
        );
    }

    rAction(a, p) {
        nestedActorAction(a, TurnOff, p);
    }
;
/* Haven't worked out the logic yet.
rPress: rStep
    name = 'rPress'
    rTrue(a, p) {
        if (!pressers.indexOf([a,p]))
            pressers += [a,p];
        return nil;
    }
    rPlans(a, p) {
        return ( [ [rBe, rReachable, p,
                    rDo, rPress, p] ] );
    }
    rAction(a, p) {
        nestedActorAction(a, Press, p);
        pressers = pressers - [a,p];
    }
    pressers = []
;
*/

/* This is for actor movement; the plan represents an actor being in a
 * location. Note that the location defines its plans for an actor
 * being in it.
 *
 * rIn means "actor is *directly* in the location."
 */
rIn: rStep
    name = 'rIn'

    rTrue(a, p) {
        return (rObjDirectlyIn.rTrue(a, [a, p]));
    }

    rPlans(a, p) {
        return (p.rMovePlan);
    }

    rAction(a, p) {
        a.scriptedTravelTo(p);
    }
;

/* We modify room, to add a couple properties we use to store
 * navigation information.
 */
modify Room
    rFromList = []  // list of directions for machine-generated
                    // movement plans created by rBuildMap function.
                    // this is only used to build the rMovePlan list.

    rMovePlan = []  // this is called by rIn, to return the moveplan
                    // for this particular parameter (location). You
                    // can override this to write custom move plans
                    // for a specific location.
;

rMapHandler: object

/* rBuildMapPlan: called by rBuildMap
*
* Modify this method to add new default move plan behavior.
* returns a valid RAP plan list for entering room from the specified
* entry (where entry is either an adjacent room, or an obstacle/doorway
* located in an adjacent room)
*
* Currently only rooms and doors, and stairways are supported.
* Other exit types will be added "soon".
*/

    rBuildMapPlan(thisRoom, thisEntry) {
        if (thisEntry.ofKind(Room)) {
            return ( [ rBe, rIn, thisEntry,
                       rDo, rIn, thisRoom ] );
        }
        if (thisEntry.ofKind(Door) && thisEntry.location) {
            return ( [ rBe, rOpen, thisEntry,
                       rBe, rIn, thisEntry.location,
                       rDo, rIn, thisRoom ] );
        }
        if (thisEntry.ofKind(Stairway) && thisEntry.location) {
            return ( [ rBe, rIn, thisEntry.location,
                       rDo, rIn, thisRoom ] );
        }
        return ([]);
    }


/* rBuildMap builds the default movement plans
 * for all RAP-enabled actors by scanning the precoded
 * exits in all rooms.
 *
 * This creates rMovePlan list for all rooms, which is
 * used by rIn. rFromList is used internally by the rBuildMap()
 * method, and is not relevant after the rMovePlan lists have
 * been set up.
 */

    rBuildMap() {
        local thisRoom, i, dest, tempList, thisEntry, thisPlan;

#ifdef __DEBUG
        "\nThe following output is for debugging and instructional
        purposes, and only prints when the game is being run in
        debug mode:\b";
        "\nrBuildMap: generating entry lists\n";

#endif

        for (thisRoom = firstObj(Room) ; thisRoom != nil ;
             thisRoom = nextObj(thisRoom, Room)) {

#ifdef __DEBUG

            "\n<<thisRoom.name>>: ";

#endif

            for (i = 1 ; i <= Direction.allDirections.length() ; i++) {

                local tmpDir = Direction.allDirections[i].dirProp;

                /* Here we're doing a lazy hack. We should be using
                 * getApparentDestination, but for the moment we
                 * have all rappers omnisciently seeing the map.
                 */
                if (thisRoom.(tmpDir))
                    dest = thisRoom.(tmpDir); // ?? .getDestination(thisRoom, nil);
                else
                    dest = nil;

                // for each valid movement, add thisRoom to the
                // destination's rFromList

                if (dest) {
                    if (dest.ofKind(Room)) {
#ifdef __DEBUG
                        "<<dest.name>> ";
#endif
                        dest.rFromList += thisRoom;
                    }
                    else if (dest.ofKind(Door) && dest.destination) {
#ifdef __DEBUG
                        "door_to <<dest.destination.name>> ";
#endif
                        dest.destination.rFromList += dest;
                    }
                    else if (dest.ofKind(Stairway) && dest.destination) {
#ifdef __DEBUG
                        "stairway_to <<dest.destination.name>> ";
#endif
                        dest.destination.rFromList += dest;
                    }
                }
            } // for this exit, i.e., dest. (i-iterator)
        }  // for thisRoom

    // for each room, construct rMovePlan from rFromList

#ifdef __DEBUG
        "\nrBuildMap: producing default plans\n";
#endif
        for (thisRoom = firstObj(Room) ; thisRoom != nil ;
              thisRoom = nextObj(thisRoom, Room)) {

            /* first we make sure the user hasn't defined a custom
             * move plan for this room.
             */
            if (thisRoom.rMovePlan == []) {
                tempList = thisRoom.rFromList;
                for (i = 1 ; i <= tempList.length() ; i++) {

                    thisEntry = tempList[i];
                    thisPlan = rBuildMapPlan(thisRoom, thisEntry);

                    thisRoom.rMovePlan += [ thisPlan ];
                }
            }

#ifdef __DEBUG
            else
                "\nCustom plan detected for:";
            "\nrBuildMap: <<thisRoom.name>> = ";
            "<<rToStr(thisRoom.rMovePlan)>>\n";
#endif
        }

    }
;

/* We're not generalizing 'p' for classes of objects, for the
 * rNotOwnedByOther, rAskFor, and rGiveTo. The parameters in these
 * cases will always involve specific objects and actors.
 */

rNotOwnedByOther: rStep
    name = 'rNotOwnedByOther'
    rTrue(a, p) {
        while (p.location) {
            if (p.location.ofKind(Actor) && p.location != a)
                return nil;
            p = p.location;
        }
        return (true);
    }
    rPlans(a, p) {
        return [];
    }
;

// note that we're not using ask-for in this version. If we were,
// we could replace the previous three lines with the following 116
/*
        local obs, act = p;
        while (!act.ofKind(Actor) && act.getOutermostRoom != self) {
            act = act.location; // act is the actor holding p
        }
        p = ((obs = act.findVisualObstructor(p)) ? obs : p);
        return( [ [rBe, rAskFor, [act, p]] ] );
    }
;

rAskFor: rStep
    name = 'rAskFor'
    rTrue(a, p) { // p = [<current possessor>, <desired object>]
        return (p[2].isIn(a));
    }
    rPlans(a, p) {
        if (p[1].location)
            return ( [ [rBe, rIn, p[1].location,
                        rDo, rAskFor, p] ] );
        return ([]);
    }
    rAction(a, p) {
        local donor, obj;
        donor = p[1];
        obj = p[2];
        new rGenericAskForTopic(a, obj, donor);

// should check if a generic topic is already defined with these params
        nestedActorAction(a, AskFor, donor, obj);

        if (donor.ofKind(Rapper))
            donor.rObjectRequest(a, obj);
    }
;

modify Rapper
    // Rapper is notified: actor wants object. Do nothing by default.

    rObjectRequest(actor, obj) { }
;

rGiveTo: rStep
    name = 'rGiveTo'
    // parameter is a two-item list, [recipient, object]
    rTrue(a, p) {
        if (dataType(p) != TypeList)
            return nil;
        return (p[2].isIn(p[1]));
    }
    rPlans(a, p) {
        if (dataType(p) != TypeList || !p[1].location)
            return [];
        return ([ [rBe, rHave, p[2],
                   rBe, rIn, p[1].location,
                   rDo, rGiveTo, p] ]);
    }
    rAction(a, p) {
        if (dataType(p) != TypeList)
            return;
// should check if a generic topic is already defined with these params
        new rGenericGiveToTopic(a, p[2], p[1]);

        nestedActorAction(a, GiveTo, p[2], p[1]);
    }
;

rGenericAskForTopic: AskForTopic
    construct(recipient, obj, donor) {
        matchObj = obj;
        location = donor;
        rMatchActor = recipient;
        initializeTopicEntry();
    }
    topicResponse {
        "\^<<gActor.name>> ask";
        if (gActor != gPlayerChar)
            "s";
        " <<location.name>> for <<matchObj.theName>>.\n";
    }
;

rGenericGiveToTopic: GiveTopic
    construct(donor, obj, recipient) {
        matchObj = obj;
        location = recipient;
        rMatchActor = donor;
        initializeTopicEntry();
    }
    topicResponse {
        "\^<<location.name>> accept";
        if (location != gPlayerChar)
            "s";
        " <<matchObj.theName>> from ";
        "<<gActor.name>>.\n";
        gDobj.moveInto(getActor);
    }
;

// these two correlate topics with actors, when rMatchActor is defined
modify TopicMatchTopic
    rMatchActor = nil
    matchTopic(fromActor, topic) {
        if(rMatchActor && fromActor != rMatchActor)
            return nil;
        return inherited(fromActor, topic);
    }
;

modify ThingMatchTopic
    rMatchActor = nil
    matchTopic(fromActor, topic) {
        if(rMatchActor && fromActor != rMatchActor)
            return nil;
        return inherited(fromActor, topic);
    }
;

*/
