"""
t3CPPG.py

Provides a class representing a constant pool page block in a t3 image file.
"""

import struct, cStringIO
from t3Block import *

class CPPG_Block(Block):
	"""
	A constant pool page block. 
	"""
	name = "Constant Pool Page Block"

	def __init__(self, datastream, image, attribs):
		Block.__init__(self, datastream, image, attribs)
		self.read_block()
		
	def read_block(self):
		"""
		Reads the data included in the block.
		"""
		id, pageidx, xor = struct.unpack("<HIB", self.datastream.read(7))
		self.data["o_cpdf"] = self.image.lastcpdf
		self.data["i_poolIdentifier"] = id
		self.data["b_bytecodePool"] = (id == 1)
		self.data["b_constantDataPool"] = (id == 2)
		self.data["i_pageIndex"] = pageidx
		self.data["xor_mask"] = xor
		self.data["s_pageData"] = self.interpret_data(xor)
				
	def interpret_data(self, mask):
		"""
		As the data of Constant Pool Pages is obscured by xor-ing each byte
		against a given mask, this method reads the data, applies the
		mask and returns the decoded data.
		"""
		if mask == 0:
			return self.datastream.read(self.i_size - 7)
		decodeddata = cStringIO.StringIO()
		for i in xrange(self.i_size - 7):
			# read the byte from our data stream 
			b = struct.unpack("<B", self.datastream.read(1))[0]
			decodeddata.write(chr(b ^ mask))
		dd = decodeddata.getvalue()
		decodeddata.close()
		return dd
		
	def report_data(self):
		sl = []
		sl.append("pool identifier: " + 
					str(self.data["i_poolIdentifier"]) + "\n")
		sl.append("bytecode pool: " + str(self.data["b_bytecodePool"]) + "\n")
		sl.append("constant pool: " + 
					str(self.data["b_constantDataPool"]) + "\n")
		sl.append("page index: " +
					str(self.data["i_pageIndex"]) + "\n" + "\n")
		return "".join(sl)